/* eslint-disable no-promise-executor-return */

import pkg from 'mkdirp'
import pkg2 from 'rimraf'

import template from './template.js'

import { basename, dirname } from 'path'
import { lstatSync, open, unlinkSync, symlinkSync } from 'fs'

import { STORE, NIXOS, GC, GCROOT, spawn, p, r, w, e, info } from './util.js'

import { fileURLToPath } from 'url'

const { sync: mkdirp } = pkg
const { sync: rimraf } = pkg2

const __dirname = dirname(fileURLToPath(import.meta.url))

function lexists (file) {
  try {
    lstatSync(file)
    return true
  } catch {
    return false
  }
}

export async function importNAR (file) {
  info('importing nar...')

  const fd = await new Promise((resolve, reject) => open(file, (err, res) => err ? reject(err) : resolve(res)))

  const { stderr, stdout, code, sig } = await spawn('nix-store', ['--import'], true, true, {
    stdio: [
      fd,
      'pipe',
      'pipe'
    ]
  })

  if (code || sig) {
    console.error(stderr) // eslint-disable-line no-console
    throw new Error(`NAR import failed with ${code || sig}`)
  }

  return stdout.split('\n').map(p => p.trim()).filter(Boolean)
}

export async function updateConfig (inputs, type) {
  info('updating config...')

  if (!e(NIXOS)) {
    mkdirp(NIXOS)
  }

  if (e(NIXOS, 'configuration.nix')) {
    rimraf(p(NIXOS, 'configuration.nix'))
  }

  if (!e(NIXOS, 'user.nix')) {
    w(NIXOS, 'user.nix', r(__dirname, 'user.nix'))
  }

  w(NIXOS, 'flake.nix', template(inputs, type))
}

export async function rebuild () {
  await spawn('nixos-rebuild', ['--flake', '/etc/nixos#system', 'switch'])
}

export async function createGCRoots (paths) {
  info('creating gc-roots...')
  rimraf(GC)
  mkdirp(GC)

  for (const [i, drv] of paths.entries()) {
    const out = p(GCROOT, `xup-${i}`)
    const mark = p(GC, basename(drv))

    if (lexists(out)) {
      unlinkSync(out)
    }

    symlinkSync(drv, mark)
    symlinkSync(mark, out)
  }
}

export async function processUpdate (yaml, nar, type, dontRebuild = false) {
  const importedPaths = await importNAR(p(STORE, 'update.nar'))

  // check missing paths

  const missing = Object.values(yaml.inputs).filter(path => !importedPaths.includes(path))

  if (missing.length > 0) {
    console.error('NAR import failed, the following paths were not found in NAR but referenced in manifest') // eslint-disable-line no-console
    for (const m of missing) console.error(' - %s', m) // eslint-disable-line no-console
  }

  await updateConfig(yaml.inputs, type)

  if (!dontRebuild) {
    await rebuild()
  }

  await createGCRoots(importedPaths)
}
