/* eslint-disable no-promise-executor-return */

import fetch from 'node-fetch'
import * as yaml from 'js-yaml'

import { createWriteStream } from 'fs'

import Joi from 'joi'

import lzma from '@xeredo/lzma-native'

import { STORE, ETC, p, r, w, e, info, error } from './util.js'
import { processUpdate } from './core.js'

const yamlValidate = Joi.object({
  version: Joi.string().required(),
  buildId: Joi.string().guid().required(),
  inputs: Joi.object().pattern(/[a-z]/, Joi.string()).required(),
  nar: Joi.object({
    source: Joi.string().uri().required(),
    compression: Joi.string()
  }).required()
}).unknown(true)

export async function downloadResource (url, path, fullUrl) {
  const res = await fetch(fullUrl ? path : `${url}/${path}`)
  if (res.status !== 200) {
    throw new Error(`Failed to fetch: ${res.status} ${res.statusText}`)
  }

  return res
}

export async function pullManifest (url) {
  const n = await (await downloadResource(url, 'update.yml')).text()
  const y = yaml.load(n)
  const { error, value } = yamlValidate.validate(y)

  if (error) {
    error('update malformed')
    throw error
  }

  return [n, value]
}

export async function checkAndPull (url, currentBuildId, onlyCheck = false) {
  info('checking for update…')

  const [n, y] = await pullManifest(url)

  const newID = y.buildId

  if ((r(ETC, 'build-id') !== newID) && (!e(STORE, 'available') || r(STORE, 'available') !== newID)) {
    if (onlyCheck) {
      return newID
    }

    info('new update available (ver %s), pulling…', y.version)

    const nar = await downloadResource(null, y.nar.source, true)

    const dest = createWriteStream(p(STORE, 'update.nar'))

    let uncomp

    switch (y.nar.compression) {
      case 'xz': {
        uncomp = nar.body.pipe(lzma.createDecompressor())
        break
      }

      case undefined: {
        uncomp = nar.body
        break
      }

      default: {
        throw new TypeError('Invalid compression ' + y.nar.compression)
      }
    }

    uncomp.pipe(dest)

    await new Promise((resolve, reject) => dest.once('close', resolve))

    w(STORE, 'update.yml', n)
    w(STORE, 'available', newID)

    info('update pulled successfully')
    return true
  }

  info('no new update available')

  if (e(STORE, 'available')) {
    info('…but already downloaded update waiting for install')
    return true
  }
}

export async function installUpdate (manifest, nar, currentVersion, type) {
  if (!e(manifest)) {
    throw new Error('no update…')
  }

  const u = r(manifest)
  const y = yaml.load(u)

  info('updating from %s to %s…', currentVersion, y.version)

  await processUpdate(y, p(nar), type)
}
