'use strict'

const ec = require('..')
const os = require('os')

const assert = require('assert').strict

const path = require('path')
const t = () => path.join(os.tmpdir(), 'ec-test-' + String(Math.random()) + '.pem')

const size = {
  rsa: 4096,
  dsa: 1024,
  ec: 128,
  ed25519: 128,
  ed448: 128,
  x25519: 128,
  x448: 128,
  dh: 128
}

const extra = {
  ec: {
    namedCurve: 'sect239k1'
  },
  dh: {
    primeLength: 12
  }
}

describe('easy-crypto', () => {
  ['rsa', 'dsa', 'ec', 'ed25519', 'ed448', 'x25519', 'x448', 'dh'].map(type => {
    describe(type, () => {
      let pair

      it('can gen a pair', async () => {
        pair = ec.generateKeyPair(type, size[type], extra[type])
      })

      it('can export & import a pair', async () => {
        assert(pair.export(ec.machineId))
        const privFile = t()
        const pubFile = t()
        pair.exportToFile(privFile, ec.machineId)

        assert(pair.pub.export())
        pair.pub.exportToFile(pubFile)

        pair = ec.private.fromFile(privFile, ec.machineId,
          ec.public.fromFile(pubFile))
      })

      it('can export & import a pair using der', async () => {
        assert(ec.private.fromDERBuffer(pair.exportDER(ec.machineId), ec.machineId))
        assert(ec.public.fromDERBuffer(pair.pub.exportDER()))
      })

      it('can encrypt, decrypt', async () => {
        const data = 'keyboardcat'
        assert(String(pair.decrypt(pair.encrypt(data))) === data)
      })

      it('can sign, verify', async () => {
        const data = 'keyboardcat'
        assert(String(pair.decrypt(pair.encrypt(data))) === data)
      })

      it('can get spki fingerprint', async () => {
        assert(pair.spkiFingerprint().length === 128)
      })
    })
  })
})
