import Boom from '@hapi/boom'
import Joi from 'joi'
import fs from 'fs'
import path from 'path'
import crypto from 'crypto'

const cookieSchema = Joi.string().required().alphanum().max(32).min(32)

export default cookieFolder => {
  function loadSession (cookie) {
    const cookieFile = path.join(cookieFolder, cookie)

    if (fs.existsSync(cookieFile)) {
      try {
        const content = String(fs.readFileSync(cookieFile))
        const [username, from, date] = content.split('\t')
        return { username, from, date }
      } catch (error) {
        return false
      }
    }
  }

  function gen () {
    return crypto.randomBytes(16).toString('hex')
  }

  return {
    authenticate (cookie, h) {
      if (!cookie) {
        throw Boom.unauthorized('No cookie supplied')
      }

      if (cookieSchema.validate(cookie).error) {
        throw Boom.unauthorized('Invalid cookie')
      }

      const session = loadSession(cookie)

      if (session) {
        return h.authenticated({ credentials: { scope: [], cookie, ...session }, artifacts: { cookie, ...session } })
      }

      throw Boom.unauthorized('Invalid cookie')
    },
    write (username, id) {
      const cookie = gen()

      fs.writeFileSync(path.join(cookieFolder, cookie), `${username}\t${id}\t${Date.now()}`)

      return cookie
    },
    gen,
    revoke (cookie) {
      const cookieFile = path.join(cookieFolder, cookie)

      if (fs.existsSync(cookieFile)) {
        fs.unlinkSync(cookieFile)
      }
    }
  }
}
