import cp from 'child_process'
import debug from 'debug'
import bl from 'bl'
import colors from 'colors/safe.js'

const log = debug('xeredo:common:utils')

export const IS_DEBUG_VM = Boolean(process.env.X_DEBUG_MODE)

export const info = Object.assign(
  function (...thing) {
    if (typeof thing[0] === 'string') {
      console.log(`${colors.grey('[*]')} ${thing[0]}`, ...thing.slice(1)) // eslint-disable-line no-console
    } else {
      console.log(...thing) // eslint-disable-line no-console
    }
  },
  {
    success (...a) {
      info(colors.green(a[0]), ...a.slice(1))
    },
    wait (...a) {
      info(colors.cyan(a[0]), ...a.slice(1))
    }
  }
)

export function error (...thing) {
  if (typeof thing[0] === 'string') {
    console.error(`${colors.red.bold('[!]')} ${colors.red(thing[0])}`, ...thing.slice(1)) // eslint-disable-line no-console
  } else {
    console.error(...thing) // eslint-disable-line no-console
  }
}

export function spawn (cmd, args, catchStdio = false, ignoreFail = false, options = {}, pipeline) {
  return new Promise((resolve, reject) => {
    const opt = [cmd, args, { stdio: (catchStdio || pipeline) ? 'pipe' : 'inherit', ...options }]
    log('spawn %o', opt)
    const p = cp.spawn(...opt)

    if (catchStdio) {
      p.stdout = p.stdout.pipe(bl())
      p.stderr = p.stderr.pipe(bl())
    } else if (pipeline) {
      p.stdout.on('data', data => pipeline.write('stdout', data))
      p.stderr.on('data', data => pipeline.write('stderr', data))
    }

    p.once('exit', (code, sig) => {
      if ((code || sig) && !ignoreFail) {
        return reject(new Error(`Failed with ${code || sig}`))
      }

      return resolve({ stdout: String(p.stdout), stderr: String(p.stderr), code, sig })
    })
  })
}

export function trErr (error, ...extra) {
  if (error.tr) {
    return [...error.tr, ...extra]
  }

  return ['common.unknown_error', String(error)]
}

export function throwTr (error, tr) {
  throw Object.assign(error, { tr })
}
