import Boom from '@hapi/boom'
import Joi from 'joi'

import SCIM from '../scim/index.js'

import debug from 'debug'

import { Issuer, generators } from 'openid-client'
import { strict as assert } from 'assert'
import { jwtVerify } from 'jose/jwt/verify'

const log = debug('common:xauth')

export default {
  pkg: {
    name: 'xauth',
    version: '1'
  },
  requirements: {
    hapi: '>=18.4.0'
  },
  register: (server, options) => {
    server.auth.scheme('xauth', implementation)
  }
}

/*

TODO:
- make this a module
- implement push logout and add parameter to session creation that contains the session id as sent from backend
- implement SIMC2 synchronisation
- implement SCIM PUSH

*/

function cmpObject (obj1, obj2) {
  try {
    assert.deepEqual(obj1, obj2)
    return true
  } catch (error) {
    return false
  }
}

const schema = Joi.object({
  openid: Joi.object({
    discoverUrl: Joi.string().uri().required(),
    clientId: Joi.string().required(),
    clientSecret: Joi.string().required(),
    callbackUrl: Joi.string().uri().required(),
    scope: Joi.string().required()
  }).required(),
  state: Joi.object({
    user: Joi.object({
      get: Joi.function().required(), // (id) => false / details
      ids: Joi.function().required(), // () => id[]
      change: Joi.function().required(), // (type=create|modify|delete, id, meta{throughSignIn, throughSimc}, details)
      mapperOIDC: Joi.function().required(), // (data) -> {id,details}
      mapperSCIM: Joi.function().required() // (data) -> {id,details}
    }).required(),
    session: Joi.object({
      validFor: Joi.number().integer().min(1000 * 60).required(),
      create: Joi.function().required(), // (data) -> id
      fetch: Joi.function().required(), // (id) -> data
      delete: Joi.function().required() // (id)
    }).required(),
    lock: Joi.object({
      aquire: Joi.function().required(), // (name, expiry)
      release: Joi.function().required() // (name)
    }).required(),
    kv: Joi.object({
      get: Joi.function().required(), // (name)
      set: Joi.function().required(), // (name, value)
      del: Joi.function().required() // (name)
    }).required()
  }).required(),
  scim: Joi.object({
    enable: Joi.boolean().default(false),
    url: Joi.string(),
    sync: Joi.boolean().default(true),
    syncInterval: Joi.number().integer().min(1000).default(1000 * 60 * 60) // 1h
  }).required()
})

const implementation = (server, options) => {
  const { value: { openid, scim, state: { user, session, lock, kv } }, error } = schema.validate(options)
  if (error) {
    throw error
  }

  const cookie = 'xsession'

  server.state(cookie, {
    encoding: 'iron',
    password: 'password-should-be-32-characters',
    path: '/',
    ttl: session.validFor,
    ignoreErrors: true,
    isSecure: false,
    isSameSite: false,
    isHttpOnly: true,
    clearInvalid: true
  })

  const res = (async function () {
    log('discovering issuer at %s', openid.discoverUrl)

    const issuer = await Issuer.discover(openid.discoverUrl)
    const client = new issuer.Client({
      client_id: openid.clientId,
      client_secret: openid.clientSecret,
      redirect_uri: openid.callbackUrl
    })

    client.verifyLogoutToken = async token => {
      const keyInput = client.id_token_signed_response_alg.startsWith('HS') ? client.client_secret : await issuer.keystore()

      // TODO: check if event type LogoutToken
      const { payload } = await jwtVerify(token, keyInput, {
        issuer: issuer.issuer,
        audience: client.client_id,
        algorithms: [client.id_token_signed_response_alg]
      })

      return payload
    }

    let grant

    // TODO: instead of fetching grant ahead of time just renew when calling auth?

    async function refreshGrant () {
      log('refreshing scim grant')
      grant = await client.grant({ grant_type: 'client_credentials' })
      const nextRefresh = (grant.expires_in - 10) * 1000
      log('next refresh %s', new Date(nextRefresh + Date.now()))
      setTimeout(refreshGrant, nextRefresh).unref()
    }

    if (scim.enable) {
      await refreshGrant()

      const scimClient = await SCIM(scim.url || `${openid.discoverUrl}/scim/v2`, () => `Bearer ${grant.access_token}`)

      await server.method({
        name: 'scimRegister',
        method: async (scimData, { h }) => {
          const suser = await scimClient.resources.User.create({
            schemas: ['urn:ietf:params:scim:schemas:core:2.0:User'],
            ...scimData
          })

          const mapped = user.mapperSCIM(suser)

          const db = await user.get(mapped.id)

          if (!db || !cmpObject(db, mapped.details)) {
            await user.change(db ? 'modify' : 'create', mapped.id, { throughSync: true }, mapped.details)
          }

          if (h) {
            // TODO: open a session for this user over openID

            const sessionId = await session.create('00', { id: mapped.id, scope: mapped.scope || [] }, new Date(Date.now() + session.validFor))

            h.state(cookie, { session: sessionId })
          }

          return mapped
        },
        options: {}
      })

      setInterval(async () => {
        log('checking sync')
        const lastSync = new Date(parseInt(await kv.get('scim') || '0', 10))
        if (Date.now() - lastSync.getTime() >= scim.syncInterval) { // if last sync is further away than syncInterval
          log('sync required, performing')
          await lock.aquire('scim', 1000 * 60) // or fail if already locked

          const ids = (await user.ids()).reduce((out, next) => {
            out[next] = true
            return out
          }, {})

          for await (const suser of (await scimClient.resources.User.get())) {
            log('process %s', suser.id)

            const mapped = user.mapperSCIM(suser)

            const db = await user.get(mapped.id)

            if (!db || !cmpObject(db, mapped.details)) {
              log('handle change %s', mapped.id)
              await user.change(db ? 'modify' : 'create', mapped.id, { throughSync: true }, mapped.details)
            }

            delete ids[mapped.id] // mark as seen
          }

          const notSeen = Object.keys(ids)
          for (let i = 0; i < notSeen.length; i++) {
            const id = notSeen[i]

            if (!await scimClient.resources.User.read(id)) { // check if they're really gone or it's just an inconsitency (index shift, etc)
              log('del %s', id)
              await user.change('delete', id, { throughSync: true })
            }
          }

          await lock.release('scim')

          await kv.set('scim', String(Date.now()))
        }
      }, scim.syncInterval).unref()
    }

    server.route({
      path: '/auth/login',
      method: 'GET',
      config: {
        auth: false,
        handler: async (request, h) => {
          const nonce = generators.nonce()

          const redirectUrl = (await client).authorizationUrl({
            scope: openid.scope,
            response_mode: 'form_post',
            nonce
          })

          h.state(cookie, { nonce })

          return h.redirect(redirectUrl)
        }
      }
    })

    server.route({
      path: '/auth/logout',
      method: 'GET',
      config: {
        auth: false,
        handler: async (request, h) => {
          if (request.state[cookie] === 'object' && request.state[cookie].session === 'string') {
            try {
              // TODO: backchannel logout to quit the entire session

              await session.del(request.state[cookie].session)
            } catch (error) {
              // fail silently as logout is already achieved by un-setting the cookie
            }
          }

          h.state(cookie)

          return h.redirect('/')
        }
      }
    })

    server.route({
      method: 'POST',
      path: new URL(openid.callbackUrl).pathname,
      config: {
        auth: false,
        handler: async (request, h) => {
          const state = request.state[cookie]

          if (typeof state !== 'object' || typeof state.nonce !== 'string') {
            throw Boom.badRequest('Invalid state')
          }

          const tokenSet = await (await client).callback(
            openid.callbackUrl, request.payload, { nonce: state.nonce }
          )

          // return

          const userInfos = await (await client).userinfo(tokenSet)

          // TODO: scim-only mode where we use scim_id claim to fetch userinfo from scim

          const mapped = user.mapperOIDC(userInfos)

          const db = await user.get(mapped.id)

          if (!db || !cmpObject(db, mapped.details)) {
            await user.change(db ? 'modify' : 'create', mapped.id, { throughSignIn: true }, mapped.details)
          }

          const sessionId = await session.create(tokenSet.claims().sid, { id: mapped.id, scope: mapped.scope || [], tokenSet }, new Date(Date.now() + session.validFor))

          h.state(cookie, { session: sessionId })

          return h.redirect('/')
        }
      }
    })

    log('xauth ready')

    return {
      authenticate: async (request, h) => {
        const state = request.state[cookie]

        if (typeof state !== 'object' || typeof state.session !== 'string') {
          throw Boom.unauthorized('Invalid state')
        }

        const sessionData = await session.fetch(state.session)
        if (!sessionData) {
          throw Boom.unauthorized('Invalid state')
        }

        const { id, scope } = sessionData
        const info = await user.get(id, true)

        return h.authenticated({
          credentials: {
            scope,
            id,
            ...info
          }
        })
      }
    }
  }())

  return {
    authenticate: async (request, h) => {
      return (await res).authenticate(request, h)
    }
  }
}
