import path from 'path'
import fs from 'fs'
import Joi from 'joi'

import { spawn } from '@xeredo/common/src/modules/util.js'
import readdir from 'recursive-readdir'

const input = {
  nginx: {
    folder: {
      'name.conf': 'config content...'
    }
  },
  firewall: [
    ['port', 'protocol or false for both', 'comment']
  ],
  domains: [
    'bla.com', 'sub1.bla.com'
  ]
}

export const NGINX_FOLDER = '/etc/nginx'

export async function cleanOldNGINXConfig () {
  const files = await readdir(NGINX_FOLDER)

  files
    .filter(f => f.endsWith('~guardian') && f.startsWith('.'))
    .map(f => [f, f.replace(/~guardian$/, '').replace(/^\./, '')])
    .forEach(([marker, actual]) => {
      fs.unlinkSync(marker)
      fs.unlinkSync(actual)
    })
}

export const format =
  Joi.object({
    nginx: Joi.object().pattern(/[a-z0-9.]/, Joi.object().pattern(/[a-z0-9.]/, Joi.string())),
    firewall: Joi.array(), // TODO: add schema [Joi.number().min(0).max(65355), Joi.string(), Joi.string()].items(),
    domains: Joi.array().items(Joi.string().domain())
  }).required()

export function merge (configs) {
  const out = {
    nginx: {},
    firewall: []
  }

  for (const source in configs) { // eslint-disable-line guard-for-in
    const value = configs[source]

    // merge nginx
    for (const folder in value.nginx || {}) { // eslint-disable-line guard-for-in
      const configs = value.nginx[folder]

      if (!out.nginx[folder]) {
        out.nginx[folder] = {}
      }

      for (const config in configs) { // eslint-disable-line guard-for-in
        const value = configs[config]

        // throw duplicate error if already exist, else add
        if (out.nginx[folder][config]) {
          throw new Error(`Duplicat key nginx->${folder}->${config} during merge detected, second key from ${source}`)
        } else {
          out.nginx[folder][config] = value
        }
      }
    }

    if (value.firewall) {
      value.firewall.forEach(([port, type, comment]) => {
        const current = out.firewall.filter(([_port]) => port === _port)[0]

        if (current) {
          // if we already have the port added

          // check if it's the same type (tcp/udp), otherwise make it both types
          if (current[1] !== type) {
            current[1] = false
          }

          // merge the comments
          current[2] += ` and ${comment}`
        } else {
          // otherwise just add
          out.firewall.push([port, type, comment])
        }
      })
    }

    if (value.domains) {
      out.domains = out.domains.concat(value.domains.filter(d => out.domains.indexOf(d) !== -1))
    }
  }

  return out
}

export async function apply (config, pipeline) {
  // TODO: firewall

  await cleanOldNGINXConfig()

  for (const folder in config.nginx) { // eslint-disable-line guard-for-in
    const contents = config.nginx[folder]

    if (!fs.existsSync(path.join(NGINX_FOLDER, folder))) {
      throw new Error(`Failed to apply config: NGINX folder ${JSON.stringify(folder)} does not exist`)
    }

    for (const file in contents) { // eslint-disable-line guard-for-in
      const config = contents[file]

      fs.writeFileSync(path.join(NGINX_FOLDER, folder, file), config)
      fs.writeFileSync(path.join(NGINX_FOLDER, folder, '.' + file + '~guardian'), 'This file is managed')
    }
  }

  // TODO: cleanup old files

  reload(pipeline)
}

export async function reload(pipeline) {
  await spawn('systemctl', ['reload', 'nginx'], false, false, {}, pipeline)
}
