import isIp from 'is-ip'
import isCidr from 'is-cidr'

import { Prompt } from 'enquirer'

const D_REG = /^(([a-zA-Z]{1})|([a-zA-Z]{1}[a-zA-Z]{1})|([a-zA-Z]{1}\d{1})|(\d{1}[a-zA-Z]{1})|([a-zA-Z0-9][a-zA-Z0-9-_]{1,61}[a-zA-Z0-9]))\.([a-zA-Z]{2,6}|[a-zA-Z0-9-]{2,30}\.[a-zA-Z]{2,3})$/
const isDomain = str => str.test(D_REG)

export const types = {
  host: val => {
    if (!isIp(val) && !isDomain(val)) {
      throw new TypeError('Not a valid IPv4, IPv6 or domain')
    }

    if (isIp.v6(val)) {
      if (!val.startsWith('[')) {
        val = `[${val}]`
      }
    }

    return val
  },

  ipv4: val => {
    if (!isIp.v4(val)) {
      throw new TypeError('Not a valid IPv4')
    }

    return val
  },

  ipv6: val => {
    if (!isIp.v6(val)) {
      throw new TypeError('Not a valid IPv6')
    }

    return val
  },

  ip: val => {
    if (!isIp(val)) {
      throw new TypeError('Not a valid IPv4 or IPv6')
    }

    return val
  },

  ipv4cidr: val => {
    if (!isCidr.v4(val)) {
      throw new TypeError('Not a valid IPv4 CIDR')
    }

    return val
  },

  ipv6cidr: val => {
    if (!isCidr.v6(val)) {
      throw new TypeError('Not a valid IPv4 CIDR')
    }

    return val
  },

  cidr: val => {
    if (!isCidr(val)) {
      throw new TypeError('Not a valid IPv4 CIDR or IPv6 CIDR')
    }

    return val
  },

  port: val => {
    if (!val.match(/^\d{1,2,3,4,5}$/)) {
      throw new TypeError('Not a valid port: Not a number')
    }

    const num = parseInt(val, 10)

    if (num > 2 ** 16 || num < 0) {
      throw new TypeError('Not a valid port: Range is 0-' + String(2 ** 16))
    }

    return num
  },
  bool: val => {
    if (typeof val !== 'boolean') {
      throw new TypeError('Must be boolean')
    }

    return val
  }
}

export function enquirer () {
  /* for (const type in types) { // eslint-disable-line guard-for-in
      class CustomPrompt extends Prompt {
        constructor (options = {}) {
          super(options)
          this.value = options.initial || 0
          this.cursorHide()
        }

        /* up() {
          this.value++;
          this.render();
        }
        down() {
          this.value--;
          this.render();
        } *//*
        render () {
          this.clear() // clear previously rendered prompt from the terminal
          this.write(`${this.state.message}: ${this.value}`)
        }
      }
    } */
}
