import _mkdirp from 'mkdirp'

import path from 'path'
import Joi from 'joi'

/*

- Most of these modules are in the wrong part (config-pregen ex)
- Most of these should be put into a "shared stuff" module that can be re-used accross apps
- The net module is the only one that follows the current interface of exposing CLI & backend functions correctly
  - If in doubt, do it like it's done in the net module and rewrite it like so

- Currently long-running stuff just "gets run like that"
  - We need to fix this, so we do it in the background
  - And respond with a websocket URL, so events can be listened to (mostly stdout)

- "yes." = fuck it, ship it :tm:

*/

import ENV from './loadenv.js'
import Verify from '../verify-serv/index.js'

import generateConfig from '../cli/config-pregen.js'
import { apply, merge } from '../config/apply.js'

import net from '@xeredo/common/src/modules/net.js'
import waf from '../waf/index.js'
import traffic from '../traffic-intercept/index.js'

import crypto from 'crypto'

import _KVBackend from '@xeredo/common/src/modules/kv/kv-json-backend.js'
import _KVStorage from '@xeredo/common/src/modules/kv/kv-storage.js'

import {
  getTypes, getType,
  getSTypes, getSType
} from '../cli/types.js'

const { sync: mkdirp } = _mkdirp

const CONFIG = '/var/lib/guardian'

const pipelineWS = ev => {
  return {
    write (stream, data) {
      ev(stream, data.toString('hex'))
    },
    log (...a) {
      ev('log', ...a)
    },
    info (...a) {
      ev('info', a)
    },
    error (...a) {
      ev('error', a)
    },
    info$t (...a) {
      ev('info$t', a)
    },
    error$t (...a) {
      ev('error$t', a)
    }
  }
}

export default (server, config) => {
  const verify = Verify({
    endpoint: ENV.VERIFY,
    signRequest: server.methods.signRequest
  })

  server.route({
    method: 'POST',
    path: '/cookie',
    config: {
      plugins: {
        licensing: {
          alwaysAllow: true
        }
      },
      handler: async (request, h) => {
        return { 'x-cli-cookie': request.auth.credentials.cookie }
      }
    }
  })

  /* Storage */

  mkdirp(CONFIG)
  const KVBackend = _KVBackend(path.join(CONFIG, 'config.json'))
  const storage = _KVStorage(KVBackend)

  server.route({
    method: 'POST',
    path: '/storage',
    config: {
      plugins: {
        licensing: {
          alwaysAllow: true
        }
      },
      validate: {
        payload: Joi.object({
          path: Joi.array().items(Joi.string()).required()
        })
      },
      handler: async (request, h) => {
        return { res: await KVBackend.get(request.payload.path) }
      }
    }
  })

  server.route({
    method: 'PUT',
    path: '/storage',
    config: {
      plugins: {
        licensing: {
          alwaysAllow: true
        }
      },
      validate: {
        payload: Joi.object({
          path: Joi.array().items(Joi.string()).required(),
          value: Joi.any().required()
        })
      },
      handler: async (request, h) => {
        return { res: await KVBackend.set(request.payload.path, request.payload.value) }
      }
    }
  })

  server.route({
    method: 'DELETE',
    path: '/storage',
    config: {
      plugins: {
        licensing: {
          alwaysAllow: true
        }
      },
      validate: {
        payload: Joi.object({
          path: Joi.array().items(Joi.string()).required()
        })
      },
      handler: async (request, h) => {
        return { res: await KVBackend.del(request.payload.path) }
      }
    }
  })

  /* Verify */

  server.route({
    method: 'POST',
    path: '/verify',
    config: {
      handler: async (request, h) => {
        let res
        const out = {
          result: {},
          success: true
        }

        const tasks = [
          verify(
            (await generateConfig(storage, { justDomainList: true })).domains
              .filter(d => d.indexOf('*') === -1)),
          generateConfig(storage, { justVerify: true })
        ]

        try {
          res = await tasks[0]
          out.result.external = res.r
          if (!res.success) {
            out.success = false
          }
        } catch (error) {
          return { error: error.status === 429 ? ['verify.info.error', ['verify.error.rate']] : ['verify.info.error', String(error)] }
        }

        try {
          res = await tasks[1]
          out.result.internal = res.result
          if (!res.success) {
            out.success = false
          }
        } catch (error) {
          return { error: ['verify.info.error', String(error)] }
        }

        return out
      }
    }
  })

  /* Net */

  server.route({
    method: 'POST',
    path: '/net/getNics',
    config: {
      plugins: {
        licensing: {
          alwaysAllow: true
        }
      },
      handler: async (request, h) => {
        return net.sys.getNics()
      }
    }
  })

  server.route({
    method: 'GET',
    path: '/net/getAddrs',
    config: {
      plugins: {
        licensing: {
          alwaysAllow: true
        }
      },
      handler: async (request, h) => {
        return net.sys.getAddrs()
      }
    }
  })

  server.route({
    method: 'POST',
    path: '/net/checkNet',
    config: {
      plugins: {
        licensing: {
          alwaysAllow: true
        }
      },
      handler: async (request, h) => {
        return net.sys.checkNet()
      }
    }
  })

  /* Config */

  // TODO: make info($t) calls in clis write pub('translate', {_:[id,params]})
  // TODO: make raw info write "log" events
  // TODO: pipe child processes to ev(log) - or copy raw I/O via "raw"

  const longs = {}

  async function long (fnc) {
    // return websocket instead of running the thing
    // and give the function some i/o sockets (or the websocket itself) for moving events arround

    const id = crypto.randomBytes(6).toString('hex')
    const url = `/long/${id}`

    longs[id] = []

    // TODO: cache the events, send new clients all the previous events
    const pub = ev => {
      // set timestamp
      ev.ts = Date.now()

      // store history
      longs[id].push(ev)

      // publish
      return server.publish(url, ev)
    };

    (async function () {
      pub({ ev: 'start' })

      try {
        const ev = (type, data) => pub({ ev: type, ...((typeof data === 'object' && !Array.isArray(data)) ? data : { data }) })
        await fnc(pipelineWS(ev), ev)
      } catch (error) {
        return pub({ ev: 'done', error: error.stack || '(invalid stacktrace)' })
      }

      return pub({ ev: 'done', error: false })
    }())

    return { id }
  }

  function idObj2List (obj, f) {
    const out = []

    if (!obj) {
      return out
    }

    for (const id in obj) { // eslint-disable-line guard-for-in
      const o = Object.assign({}, obj[id])
      o.id = id
      out.push(o)

      if (f) {
        f(o)
      }
    }

    return out
  }

  function list2IdObj (list, f) {
    if (!list || !list.length) {
      return {}
    }

    return list.reduce((out, next) => {
      out[next.id] = next
      if (f) {
        f(next)
      }
      delete next.id

      return out
    }, {})
  }

  const ID = Joi.string().required().regex(/[a-z][a-z0-9]+/i)

  server.subscription('/long/{id}', {
    onSubscribe (socket, path) {
      const id = path.split('/').pop()

      if (!longs[id]) {
        return socket.revoke(path, { ev: '404' })
      }

      // publish previous events
      longs[id].forEach(ev => socket.publish(path, ev))

      // client is subscribed now, will recive events live
    }
  })

  server.route({
    method: 'GET',
    path: '/config/guardian',
    config: {
      handler: async (request, h) => {
        const current = (await (storage.sub(['guardian']).get())) || {}

        const describe = await generateConfig(storage, { justDescribe: true })

        return {
          domains: idObj2List(current.domains, domain => {
            domain.routes = idObj2List(domain.routes, route => {
              route.describe = describe[route.id]
            })
          }),
          services: idObj2List(current.services, service => {
            service.describe = describe[service.id]
          })
        }
      }
    }
  })

  server.route({
    method: 'GET',
    path: '/config/guardian/plugin-info',
    config: {
      handler: async (request, h) => {
        return {
          route: getTypes().reduce((out, plugin) => {
            const type = getType(plugin)
            out[plugin] = {
              name: type.name,
              desc: type.desc,
              locales: type.locales,
              config: type.config
            }

            return out
          }, {}),
          service: getSTypes().reduce((out, plugin) => {
            const type = getSType(plugin)
            out[plugin] = {
              name: type.name,
              desc: type.desc,
              locales: type.locales,
              config: type.config
            }

            return out
          }, {})

        }
      }
    }
  })

  server.route({
    method: 'POST',
    path: '/config/guardian',
    config: {
      validate: {
        payload: Joi.object({
          domains: Joi.array().required().items(Joi.object({
            id: ID,
            name: Joi.string().domain().required(),
            useMain: Joi.boolean(),
            routes: Joi.array().required().items(Joi.object({
              id: ID,
              type: Joi.string().required(),
              dest: Joi.string(),
              describe: Joi.any(),
              config: Joi.object().required()
            }))
          })),
          services: Joi.array().required().items(Joi.object({
            id: ID,
            type: Joi.string().required(),
            describe: Joi.any(),
            config: Joi.object().required()
          }))
        })
      },
      handler: async (request, h) => {
        await (storage.sub(['guardian'])).set({
          domains: list2IdObj(request.payload.domains, domain => {
            domain.routes = list2IdObj(domain.routes, route => {
              delete route.describe
            })
          }),
          services: list2IdObj(request.payload.services, service => {
            delete service.describe
          })
        })

        return { ok: true }
      }
    }
  })

  server.route({
    method: 'POST',
    path: '/config/guardian/apply',
    config: {
      handler: async (request, h) => {
        return long(async pipeline => {
          const generated = await generateConfig(storage, { pipeline })
          await apply(merge(generated), pipeline)
        })
      }
    }
  })

  server.route({
    method: 'POST',
    path: '/config/network',
    config: {
      plugins: {
        licensing: {
          alwaysAllow: true
        }
      },
      handler: async (request, h) => {
        return long(async pipeline => {
          await net.sys.setupNic(storage, pipeline)
        })
      }
    }
  })

  /* WAF */

  server.route({
    method: 'GET',
    path: '/config/waf',
    config: {
      handler: async (request, h) => {
        return waf.sys.getWAFList(storage)
      }
    }
  })

  server.route({
    method: 'POST',
    path: '/config/waf',
    config: {
      validate: {
        payload: Joi.object({
          type: Joi.string(),
          enabled: Joi.boolean()
        })
      },
      handler: async (request, h) => {
        for (const key in request.payload) { // eslint-disable-line guard-for-in
          await (storage.sub(['guardian', 'waf', key]).set(request.payload[key]))
        }

        return long(async pipeline => {
          await waf.sys.setupWAF(storage, pipeline)
        })
      }
    }
  })

  server.route({
    method: 'GET',
    path: '/config/traffic/keylog',
    config: {
      handler: async (request, h) => {
        return traffic.sys.getKeylogInfo(storage)
      }
    }
  })

  server.route({
    method: 'POST',
    path: '/config/traffic/keylog',
    config: {
      handler: async (request, h) => {
        await traffic.sys.enableKeylog(storage)
        return traffic.sys.getKeylogInfo(storage)
      }
    }
  })

  server.route({
    method: 'DELETE',
    path: '/config/traffic/keylog',
    config: {
      handler: async (request, h) => {
        await traffic.sys.disableKeylog(storage)
        return traffic.sys.getKeylogInfo(storage)
      }
    }
  })

  server.route({
    method: 'GET',
    path: '/config/traffic/pcap',
    config: {
      handler: async (request, h) => {
        return traffic.sys.getPCAPInfo(storage)
      }
    }
  })

  server.route({
    method: 'POST',
    path: '/config/traffic/pcap',
    config: {
      handler: async (request, h) => {
        await traffic.sys.enablePCAP(storage)
        return traffic.sys.getPCAPInfo(storage)
      }
    }
  })

  server.route({
    method: 'DELETE',
    path: '/config/traffic/pcap',
    config: {
      handler: async (request, h) => {
        await traffic.sys.disablePCAP(storage)
        return traffic.sys.getPCAPInfo(storage)
      }
    }
  })

  setInterval(async () => { // daily cert renew & waf renew cron
    return long(async pipeline => {
      await waf.sys.setupWAF(storage, pipeline)
      await generateConfig(storage, { pipeline })
    })
  }, 1000 * 60 * 60 * 24).unref()
}
