import ec from '@xeredo/easy-crypto'

import fs from 'fs'
import path from 'path'
import Auth from '@xeredo/crypto-auth/src/client.js'

import protons from 'protons'
import { fileURLToPath } from 'url'

const __dirname = path.dirname(fileURLToPath(import.meta.url))
const { LCertContainer, LCert, BCertContainer, BCert } = protons(fs.readFileSync(path.join(__dirname, 'licensing.proto')))

export default ({ storage, serverKey, bcert }) => {
  const p = (...a) => path.join(storage, ...a)

  let self

  if (fs.existsSync(p('private.pem'))) {
    self = ec.private.fromFile(p('private.pem'), ec.machineId,
      ec.public.fromFile(p('public.pem')))
  } else {
    self = ec.generateKeyPair('rsa', 4096)
    self.pub.exportToFile(p('public.pem'))
    self.exportToFile(p('private.pem'), ec.machineId)
  }

  const client = Auth(self)

  let keys
  let currentLCert
  let currentBCert

  function loadLCert (lcert) {
    const cont = LCertContainer.decode(lcert)
    const certBinary = LCert.encode(LCert.decode(LCert.encode(cont.cert)))

    if (cont.expires < Date.now()) {
      throw new Error('LCert process failed: Expired')
    }

    const key = keys[cont.key.toString('hex')]

    if (!key) {
      throw new Error('LCert process failed: Key unknown')
    }

    if (!key.verify(certBinary, cont.signature)) {
      throw new Error('LCert process failed: Invalid signature')
    }

    return cont.cert
  }

  function loadBCert (bcert, unsafe = false) {
    const cont = BCertContainer.decode(bcert)

    if (unsafe) {
      return cont.cert
    }

    const certBinary = BCert.encode(BCert.decode(BCert.encode(cont.cert)))

    const commonKey = Object.keys(keys)
      .filter(id => cont.sigs.filter(s => s.key.toString('hex') === id))[0]

    if (!commonKey) {
      throw new Error('BCert process failed: no common key')
    }

    if (cont.expires < Date.now()) {
      throw new Error('LCert process failed: Expired')
    }

    const key = keys[commonKey]

    if (!key) {
      throw new Error('BCert process failed: Key unknown')
    }

    if (!key.verify(certBinary, cont.sigs.filter(s => s.key.toString('hex') === commonKey)[0].signature)) {
      throw new Error('BCert process failed: Invalid signature')
    }

    return cont.cert
  }

  function bcertKeys () {
    keys = currentBCert.keys.reduce((out, key) => {
      const pub = ec.public.fromDERBuffer(key)
      const fingerprint = pub.spkiFingerprint()

      out[fingerprint] = pub

      return out
    }, {})
  }

  currentBCert = loadBCert(fs.readFileSync(bcert), true)
  bcertKeys()

  if (fs.existsSync(p('bcert'))) {
    try {
      currentBCert = loadBCert(fs.readFileSync(p('bcert')))
      bcertKeys()
    } catch (error) {
      // ignore
    }
  }

  if (fs.existsSync(p('lcert'))) {
    try {
      currentLCert = loadLCert(fs.readFileSync(p('lcert')))
    } catch (error) {
      // ignore
    }
  }

  return {
    self,
    signedRequest: (url, method = 'GET', payload) => {
      return client.signFetch(url, method, payload)
    },
    processLCert (lcert) {
      lcert = Buffer.from(lcert, 'base64')

      currentLCert = loadLCert(lcert)

      fs.writeFileSync(p('lcert'), lcert)
    },
    processBCert (bcert) {
      bcert = Buffer.from(bcert, 'base64')

      currentBCert = loadBCert(bcert)
      bcertKeys()

      fs.writeFileSync(p('bcert'), bcert)
    },
    get keys () {
      return keys
    },
    get lcert () {
      return currentLCert
    },
    key (id) {
      return keys[id]
    }
  }
}
