import net from 'net'
import fetch from 'node-fetch'

import http from 'http'
import https from 'https'

import debug from 'debug'
const log = debug('guardian:verify')

export const Errors = {
  SUCCESS: 0,

  DNS_NO_RESOLVE: 10,
  DNS_OTHER: 19,

  CONN_REFUSED: 20,
  CONN_TIMEOUT: 21,
  CONN_OTHER: 29,

  HTTP_OTHER: 39,

  STATUS_NON_200: 40, // any above 400
  STATUS_404: 41,
  STATUS_500: 42, // any 500
  STATUS_502: 43,
  STATUS_503: 44,

  OTHER: 99
}

export const ErrorToTranslated = Object.keys(Errors).map(key => [Errors[key], `verify.result.${key.toLowerCase()}`]).reduce((out, next) => {
  out[next[0]] = next[1]

  return out
}, {})

function verifyNetReal (host, port, opts = {}, netOpts = {}) {
  return new Promise((resolve, reject) => {
    const { timeout = 5000 } = opts

    const socket = new net.Socket()

    socket.setTimeout(timeout)

    socket.once('error', error => {
      log('conn error %o', error)

      switch (error.code) {
        case 'ECONNREFUSED': return reject(Errors.CONN_REFUSED)
        case 'ETIMEDOUT': return reject(Errors.CONN_TIMEOUT)
        default: return reject(Errors.CONN_OTHER)
      }
    })

    socket.once('timeout', () => {
      log('conn timeout')
      socket.destroy()

      return reject(Errors.CONN_TIMEOUT)
    })

    return socket.connect({
      host,
      port: parseInt(port, 10)
    }, () => {
      socket.destroy()
      resolve()
    })
  })
}

const httpAgent = new http.Agent({})
const httpsAgent = new https.Agent({
  rejectUnauthorized: false
})

async function verifyURLReal (url, opts = {}, httpOpts = {}) {
  const { timeout = 5000 } = opts

  log('timeout', timeout)

  const controller = new AbortController()
  const abortTimeout = setTimeout(() => controller.abort(), timeout).unref()

  try {
    const res = await fetch(url, {
      // timeout,
      signal: controller.signal,
      agent: function (_parsedURL) {
        if (_parsedURL.protocol === 'http:') {
          return httpAgent
        }

        return httpsAgent
      },
      ...httpOpts
    })
    clearTimeout(abortTimeout)

    switch (true) {
      case res.status <= 399: return
      case res.status === 503: throw Errors.STATUS_503
      case res.status === 502: throw Errors.STATUS_502
      case res.status >= 500 && res.status <= 599: throw Errors.STATUS_500
      case res.status === 404: throw Errors.STATUS_404
      default: throw Errors.STATUS_NON_200
    }
  } catch (error) {
    if (typeof error === 'number') {
      throw error // just re-throw
    }

    switch (error.code || error.name) {
      case 'ECONNREFUSED': throw Errors.CONN_REFUSED
      case 'ETIMEDOUT': throw Errors.CONN_TIMEOUT
      case 'AbortError': throw Errors.CONN_TIMEOUT
      default: throw error.code && error.code.startsWith('ECONN') ? Errors.CONN_OTHER : Errors.OTHER
    }
  }
}

async function verifyPost (promise, itemDisplay) {
  log('verify %s', itemDisplay)

  try {
    await promise

    log('done verify %s, success', itemDisplay)

    return {
      state: 'success',
      success: true,
      error: 0,
      errorDisplay: [],
      display: ['verify.item', itemDisplay, ['verify.summary.reachable']]
    }
  } catch (error) {
    if (typeof error !== 'number') {
      log('other error %o', error)
      error = Errors.OTHER // eslint-disable-line no-ex-assign
    }

    const errorDisplay = [ErrorToTranslated[error]]

    log('done verify %s, error %s', itemDisplay, errorDisplay)

    return {
      state: 'failure',
      success: false,
      error,
      errorDisplay,
      display: ['verify.item', itemDisplay, ['verify.summary.not_reachable_desc', errorDisplay]]
    }
  }
}

export function verifyNet (host, port, opts, netOpts) {
  return verifyPost(verifyNetReal(host, port, opts, netOpts), `${host}:${port}`)
}

export function verifyURL (url, opts, httpOpts, netOpts) {
  return verifyPost(verifyURLReal(url, opts, httpOpts, netOpts), url)
}
