# Email setup is fairly involved, useful references:
# https://drewdevault.com/2018/08/05/Local-mail-server.html

{ config, lib, pkgs, ... }:

with lib;
let
  cfg = config.services.sourcehut;
  cfgIni = cfg.settings;
  scfg = cfg.lists;
  iniKey = "lists.sr.ht";

  rcfg = config.services.redis;
  drv = pkgs.sourcehut.listssrht;
in
{
  options.services.sourcehut.lists = {
    user = mkOption {
      type = types.str;
      default = "listssrht";
      description = ''
        User for lists.sr.ht.
      '';
    };

    port = mkOption {
      type = types.port;
      default = 5006;
      description = ''
        Port on which the "lists" module should listen.
      '';
    };

    database = mkOption {
      type = types.str;
      default = "lists.sr.ht";
      description = ''
        PostgreSQL database name for lists.sr.ht.
      '';
    };

    statePath = mkOption {
      type = types.path;
      default = "${cfg.statePath}/listssrht";
      description = ''
        State path for lists.sr.ht.
      '';
    };
  };

  config = with scfg; lib.mkIf (cfg.enable && elem "lists" cfg.services) {
    users = {
      users = {
        "${user}" = {
          isSystemUser = true;
          group = user;
          extraGroups = [ "postfix" ];
          description = "lists.sr.ht user";
        };
      };
      groups = {
        "${user}" = { };
      };
    };

    services.postgresql = {
      authentication = ''
        local ${database} ${user} trust
      '';
      ensureDatabases = [ database ];
      ensureUsers = [
        {
          name = user;
          ensurePermissions = { "DATABASE \"${database}\"" = "ALL PRIVILEGES"; };
        }
      ];
    };

    systemd = {
      tmpfiles.rules = [
        "d ${statePath} 0750 ${user} ${user} -"
      ];

      services = {
        listssrht = import ./service.nix { inherit config pkgs lib; } scfg drv iniKey {
          after = [ "postgresql.service" "network.target" ];
          requires = [ "postgresql.service" ];
          wantedBy = [ "multi-user.target" ];

          description = "lists.sr.ht website service";

          serviceConfig.ExecStart = "${cfg.python}/bin/gunicorn ${drv.pname}.app:app -b ${cfg.address}:${toString port}";
        };

        listssrht-process = {
          after = [ "postgresql.service" "network.target" ];
          requires = [ "postgresql.service" ];
          wantedBy = [ "multi-user.target" ];

          description = "lists.sr.ht process service";
          serviceConfig = {
            Type = "simple";
            User = user;
            Restart = "always";
            ExecStart = "${cfg.python}/bin/celery -A ${drv.pname}.process worker --loglevel=info";
          };
        };

        listssrht-lmtp = {
          after = [ "postgresql.service" "network.target" ];
          requires = [ "postgresql.service" ];
          wantedBy = [ "multi-user.target" ];

          description = "lists.sr.ht process service";
          serviceConfig = {
            Type = "simple";
            User = user;
            Restart = "always";
            ExecStart = "${cfg.python}/bin/listssrht-lmtp";
          };
        };


        listssrht-webhooks = {
          after = [ "postgresql.service" "network.target" ];
          requires = [ "postgresql.service" ];
          wantedBy = [ "multi-user.target" ];

          description = "lists.sr.ht webhooks service";
          serviceConfig = {
            Type = "simple";
            User = user;
            Restart = "always";
            ExecStart = "${cfg.python}/bin/celery -A ${drv.pname}.webhooks worker --loglevel=info";
          };
        };
      };
    };

    services.sourcehut.settings = {
      # URL lists.sr.ht is being served at (protocol://domain)
      "lists.sr.ht".origin = mkDefault "http://lists.${cfg.originBase}";
      # Address and port to bind the debug server to
      "lists.sr.ht".debug-host = mkDefault "0.0.0.0";
      "lists.sr.ht".debug-port = mkDefault port;
      # Configures the SQLAlchemy connection string for the database.
      "lists.sr.ht".connection-string = mkDefault "postgresql:///${database}?user=${user}&host=/var/run/postgresql";
      # Set to "yes" to automatically run migrations on package upgrade.
      "lists.sr.ht".migrate-on-upgrade = mkDefault "yes";
      # lists.sr.ht's OAuth client ID and secret for meta.sr.ht
      # Register your client at meta.example.org/oauth
      "lists.sr.ht".oauth-client-id = mkDefault null;
      "lists.sr.ht".oauth-client-secret = mkDefault null;
      # Outgoing email for notifications generated by users
      "lists.sr.ht".notify-from = mkDefault "CHANGEME@example.org";
      # The redis connection used for the webhooks worker
      "lists.sr.ht".webhooks = mkDefault "redis://${rcfg.bind}:${toString rcfg.port}/2";
      # The redis connection used for the celery worker
      "lists.sr.ht".redis = mkDefault "redis://${rcfg.bind}:${toString rcfg.port}/4";
      # Network-key
      "lists.sr.ht".network-key = mkDefault null;
      # Allow creation
      "lists.sr.ht".allow-new-lists = mkDefault "no";
      # Posting Domain
      "lists.sr.ht".posting-domain = mkDefault "lists.${cfg.originBase}";

      # Path for the lmtp daemon's unix socket. Direct incoming mail to this socket.
      # Alternatively, specify IP:PORT and an SMTP server will be run instead.
      "lists.sr.ht::worker".sock = mkDefault "/tmp/lists.sr.ht-lmtp.sock";
      # The lmtp daemon will make the unix socket group-read/write for users in this
      # group.
      "lists.sr.ht::worker".sock-group = mkDefault "postfix";
      "lists.sr.ht::worker".reject-url = mkDefault "https://man.sr.ht/lists.sr.ht/etiquette.md";
      "lists.sr.ht::worker".reject-mimetypes = mkDefault "text/html";

    };

    services.nginx.virtualHosts."lists.${cfg.originBase}" = {
      forceSSL = true;
      locations."/".proxyPass = "http://${cfg.address}:${toString port}";
      locations."/query".proxyPass = "http://${cfg.address}:${toString (port + 100)}";
      locations."/static".root = "${pkgs.sourcehut.listssrht}/${pkgs.sourcehut.python.sitePackages}/listssrht";
    };
  };
}
