{ stdenv, symlinkJoin, lib, makeWrapper
, writeText
, bundlerEnv, ruby
, nodejs
, nodePackages
, python3Packages
, callPackage
}:
with lib;

neovim:

let
  wrapper = {
      extraName ? ""
    # should contain all args but the binary. Can be either a string or list
    , wrapperArgs ? []
    , manifestRc ? null
    , withPython2 ? false
    , withPython3 ? true,  python3Env ? null
    , withNodeJs ? false
    , rubyEnv ? null
    , vimAlias ? false
    , viAlias ? false

    # additional argument not generated by makeNeovimConfig
    # it will append "-u <customRc>" to the wrapped arguments
    # set to false if you want to control where to save the generated config
    # (e.g., in ~/.config/init.vim or project/.nvimrc)
    , wrapRc ? true
    , neovimRcContent ? ""
    , ...
  }@args:
  let

    wrapperArgsStr = if isString wrapperArgs then wrapperArgs else lib.escapeShellArgs wrapperArgs;

    # If configure != {}, we can't generate the rplugin.vim file with e.g
    # NVIM_SYSTEM_RPLUGIN_MANIFEST *and* NVIM_RPLUGIN_MANIFEST env vars set in
    # the wrapper. That's why only when configure != {} (tested both here and
    # when postBuild is evaluated), we call makeWrapper once to generate a
    # wrapper with most arguments we need, excluding those that cause problems to
    # generate rplugin.vim, but still required for the final wrapper.
    finalMakeWrapperArgs =
      [ "${neovim}/bin/nvim" "${placeholder "out"}/bin/nvim" ]
      ++ [ "--set" "NVIM_SYSTEM_RPLUGIN_MANIFEST" "${placeholder "out"}/rplugin.vim" ]
      ++ optionals wrapRc [ "--add-flags" "-u ${writeText "init.vim" neovimRcContent}" ]
      ;
  in
  assert withPython2 -> throw "Python2 support has been removed from the neovim wrapper, please remove withPython2 and python2Env.";

  symlinkJoin {
      name = "neovim-${lib.getVersion neovim}${extraName}";
      # Remove the symlinks created by symlinkJoin which we need to perform
      # extra actions upon
      postBuild = lib.optionalString stdenv.isLinux ''
        rm $out/share/applications/nvim.desktop
        substitute ${neovim}/share/applications/nvim.desktop $out/share/applications/nvim.desktop \
          --replace 'Name=Neovim' 'Name=WrappedNeovim'
      ''
      + optionalString withPython3 ''
        makeWrapper ${python3Env}/bin/python3 $out/bin/nvim-python3 --unset PYTHONPATH
      ''
      + optionalString (rubyEnv != null) ''
        ln -s ${rubyEnv}/bin/neovim-ruby-host $out/bin/nvim-ruby
      ''
      + optionalString withNodeJs ''
        ln -s ${nodePackages.neovim}/bin/neovim-node-host $out/bin/nvim-node
      ''
      + optionalString vimAlias ''
        ln -s $out/bin/nvim $out/bin/vim
      ''
      + optionalString viAlias ''
        ln -s $out/bin/nvim $out/bin/vi
      ''
      + optionalString (manifestRc != null) (let
        manifestWrapperArgs =
          [ "${neovim}/bin/nvim" "${placeholder "out"}/bin/nvim-wrapper" ];
      in ''
        echo "Generating remote plugin manifest"
        export NVIM_RPLUGIN_MANIFEST=$out/rplugin.vim
        makeWrapper ${lib.escapeShellArgs manifestWrapperArgs} ${wrapperArgsStr}

        # Some plugins assume that the home directory is accessible for
        # initializing caches, temporary files, etc. Even if the plugin isn't
        # actively used, it may throw an error as soon as Neovim is launched
        # (e.g., inside an autoload script), causing manifest generation to
        # fail. Therefore, let's create a fake home directory before generating
        # the manifest, just to satisfy the needs of these plugins.
        #
        # See https://github.com/Yggdroot/LeaderF/blob/v1.21/autoload/lfMru.vim#L10
        # for an example of this behavior.
        export HOME="$(mktemp -d)"
        # Launch neovim with a vimrc file containing only the generated plugin
        # code. Pass various flags to disable temp file generation
        # (swap/viminfo) and redirect errors to stderr.
        # Only display the log on error since it will contain a few normally
        # irrelevant messages.
        if ! $out/bin/nvim-wrapper \
          -u ${writeText "manifest.vim" manifestRc} \
          -i NONE -n \
          -E -V1rplugins.log -s \
          +UpdateRemotePlugins +quit! > outfile 2>&1; then
          cat outfile
          echo -e "\nGenerating rplugin.vim failed!"
          exit 1
        fi
        rm "${placeholder "out"}/bin/nvim-wrapper"
      '')
      + ''
        rm $out/bin/nvim
        touch $out/rplugin.vim
        makeWrapper ${lib.escapeShellArgs finalMakeWrapperArgs} ${wrapperArgsStr}
      '';

    paths = [ neovim ];

    preferLocalBuild = true;

    nativeBuildInputs = [ makeWrapper ];
    passthru = {
      unwrapped = neovim;
      initRc = neovimRcContent;

      tests = callPackage ./tests.nix {
      };
    };

    meta = neovim.meta // {
      # To prevent builds on hydra
      hydraPlatforms = [];
      # prefer wrapper over the package
      priority = (neovim.meta.priority or 0) - 1;
    };
  };
in
  lib.makeOverridable wrapper
