#!/usr/bin/env bash
set -e

URL=https://github.com/tootsuite/mastodon.git

POSITIONAL=()
while [[ $# -gt 0 ]]; do
    key="$1"

    case $key in
        --url)
        URL="$2"
        shift # past argument
        shift # past value
        ;;
        --ver)
        VERSION="$2"
        shift # past argument
        shift # past value
        ;;
    --rev)
    REVISION="$2"
        shift # past argument
        shift # past value
        ;;
    --patches)
    PATCHES="$2"
        shift # past argument
        shift # past value
        ;;
        *)    # unknown option
        POSITIONAL+=("$1")
        shift # past argument
        ;;
    esac
done

if [[ -z "$VERSION" || -n "$POSITIONAL" ]]; then
    echo "Usage: update.sh [--url URL] --ver VERSION [--rev REVISION] [--patches PATCHES]"
    echo "URL may be any path acceptable to 'git clone' and VERSION the"
    echo "semantic version number.  If VERSION is not a revision acceptable to"
    echo "'git checkout', you must provide one in REVISION.  If URL is not"
    echo "provided, it defaults to https://github.com/tootsuite/mastodon.git."
    echo "PATCHES, if provided, should be one or more Nix expressions"
    echo "separated by spaces."
    exit 1
fi

if [[ -z "$REVISION" ]]; then
    REVISION="$VERSION"
fi

rm -f gemset.nix yarn.nix version.nix version.patch source.nix package.json
TARGET_DIR="$PWD"


WORK_DIR=$(mktemp -d)

# Check that working directory was created.
if [[ ! "$WORK_DIR" || ! -d "$WORK_DIR" ]]; then
    echo "Could not create temporary directory"
    exit 1
fi

# Delete the working directory on exit.
function cleanup {
    # Report errors, if any, from nix-prefetch-git
    grep "fatal" $WORK_DIR/nix-prefetch-git.out >/dev/stderr || true
    rm -rf "$WORK_DIR"
}
trap cleanup EXIT

echo "Fetching source code $REVISION from $URL"
JSON=$(nix-prefetch-git --url "$URL" --rev "$REVISION"  2> $WORK_DIR/nix-prefetch-git.out)
SHA=$(echo $JSON | jq -r .sha256)
FETCHED_SOURCE_DIR=$(grep '^path is' $WORK_DIR/nix-prefetch-git.out | sed 's/^path is //')

echo "Creating version.nix"
echo \"$VERSION\" | sed 's/^"v/"/' > version.nix

echo "Creating source.nix"
# yarn2nix and mkYarnPackage want the version to be present in
# package.json. Mastodon itself does not include the version in
# package.json but at least one fork (Soapbox) does.
if [ $(jq .version $FETCHED_SOURCE_DIR/package.json) == "null" ]; then
    mkdir $WORK_DIR/a $WORK_DIR/b
    cp $FETCHED_SOURCE_DIR/package.json $WORK_DIR/a
    cd $WORK_DIR
    jq "{version:$(cat $TARGET_DIR/version.nix)} + ." a/package.json > b/package.json
    diff -Naur --label a/package.json --label b/package.json a b > $TARGET_DIR/version.patch || true
    rm -rf a b tmp
    cd $TARGET_DIR
    PATCHES="$PATCHES ./version.patch "
fi

cat > source.nix << EOF
# This file was generated by pkgs.mastodon.updateScript.
{ fetchgit, applyPatches }: let
  src = fetchgit {
    url = "$URL";
    rev = "$REVISION";
    sha256 = "$SHA";
  };
in applyPatches {
  inherit src;
  patches = [$PATCHES];
}
EOF
SOURCE_DIR="$(nix-build --no-out-link -E '(import <nixpkgs> {}).callPackage ./source.nix {}')"

echo "Creating gemset.nix"
bundix --lockfile="$SOURCE_DIR/Gemfile.lock" --gemfile="$SOURCE_DIR/Gemfile"
echo "" >> $TARGET_DIR/gemset.nix  # Create trailing newline to please EditorConfig checks

echo "Creating yarn.nix"
cp -r $SOURCE_DIR/* $WORK_DIR
chmod -R u+w $WORK_DIR
cd $WORK_DIR
yarn2nix > $TARGET_DIR/yarn.nix
sed "s/https___.*_//g" -i $TARGET_DIR/yarn.nix
cp $WORK_DIR/package.json $TARGET_DIR
