{ lib
, multiStdenv
, fetchFromGitHub
, fetchpatch
, substituteAll
, meson
, ninja
, pkg-config
, wine
, boost
, libxcb
, pkgsi686Linux
}:

let
  # Derived from subprojects/bitsery.wrap
  bitsery = rec {
    version = "5.2.0";
    src = fetchFromGitHub {
      owner = "fraillt";
      repo = "bitsery";
      rev = "v${version}";
      hash = "sha256-Bxdtjn2v2lP2lCnvjzmct6QHT7FpwmXoSZtd2oEFS4w=";
    };
  };

  # Derived from subprojects/function2.wrap
  function2 = rec {
    version = "4.1.0";
    src = fetchFromGitHub {
      owner = "Naios";
      repo = "function2";
      rev = version;
      hash = "sha256-JceZU8ZvtYhFheh8BjMvjjZty4hcYxHEK+IIo5X4eSk=";
    };
  };

  # Derived from subprojects/tomlplusplus.wrap
  tomlplusplus = rec {
    version = "2.1.0";
    src = fetchFromGitHub {
      owner = "marzer";
      repo = "tomlplusplus";
      rev = "v${version}";
      hash = "sha256-i6yAEqwkinkPEzzb6ynXytS1SEOUDwi8SixMf62NVzs=";
    };
  };

  # Derived from vst3.wrap
  vst3 = rec {
    version = "3.7.3_build_20-patched";
    src = fetchFromGitHub {
      owner = "robbert-vdh";
      repo = "vst3sdk";
      rev = "v${version}";
      fetchSubmodules = true;
      sha256 = "sha256-m2y7No7BNbIjLNgdAqIAEr6UuAZZ/wwM2+iPWKK17gQ=";
    };
  };
in multiStdenv.mkDerivation rec {
  pname = "yabridge";
  version = "3.6.0";

  # NOTE: Also update yabridgectl's cargoHash when this is updated
  src = fetchFromGitHub {
    owner = "robbert-vdh";
    repo = pname;
    rev = version;
    hash = "sha256-lgSkZ0i2DojP6HXJP3cC5FUtfv7R/nsSiHT60bPSyLc=";
  };

  # Unpack subproject sources
  postUnpack = ''(
    cd "$sourceRoot/subprojects"
    cp -R --no-preserve=mode,ownership ${bitsery.src} bitsery-${bitsery.version}
    tar -xf bitsery-patch-${bitsery.version}.tar.xz
    cp -R --no-preserve=mode,ownership ${function2.src} function2-${function2.version}
    tar -xf function2-patch-${function2.version}.tar.xz
    cp -R --no-preserve=mode,ownership ${tomlplusplus.src} tomlplusplus
    cp -R --no-preserve=mode,ownership ${vst3.src} vst3
  )'';

  patches = [
    # Hard code wine path so wine version is correct in logs
    (substituteAll {
      src = ./hardcode-wine.patch;
      inherit wine;
    })
    # Remove with next yabridge update
    (fetchpatch {
      name = "fix-for-wine-6.20.patch";
      url = "https://github.com/robbert-vdh/yabridge/commit/5be149cb525a638f7fc3adf84918c8239ee50ecf.patch";
      sha256 = "sha256-x/gfn4mKZIGQ4M0o/0LlZF8i8wZDx/bkwf8wp0BGDBo=";
    })
  ];

  postPatch = ''
    patchShebangs .
  '';

  nativeBuildInputs = [
    meson
    ninja
    pkg-config
    wine
  ];

  buildInputs = [
    boost
    libxcb
  ];

  # Meson is no longer able to pick up Boost automatically.
  # https://github.com/NixOS/nixpkgs/issues/86131
  BOOST_INCLUDEDIR = "${lib.getDev boost}/include";
  BOOST_LIBRARYDIR = "${lib.getLib boost}/lib";

  mesonFlags = [
    "--cross-file" "cross-wine.conf"
    "-Dwith-bitbridge=true"

    # Requires CMake and is unnecessary
    "-Dtomlplusplus:GENERATE_CMAKE_CONFIG=disabled"

    # tomlplusplus examples and tests don't build with winegcc
    "-Dtomlplusplus:BUILD_EXAMPLES=disabled"
    "-Dtomlplusplus:BUILD_TESTS=disabled"
  ];

  preConfigure = ''
    sed -i "221s|xcb.*|xcb_32bit_dep = winegcc.find_library('xcb', dirs: [ '${lib.getLib pkgsi686Linux.xorg.libxcb}/lib', ])|" meson.build
    sed -i "199 i '${lib.getLib pkgsi686Linux.boost}/lib'," meson.build
  '';

  installPhase = ''
    runHook preInstall
    mkdir -p "$out/bin" "$out/lib"
    cp yabridge-group*.exe{,.so} "$out/bin"
    cp yabridge-host*.exe{,.so} "$out/bin"
    cp libyabridge-vst2.so "$out/lib"
    cp libyabridge-vst3.so "$out/lib"
    runHook postInstall
  '';

  # Hard code wine path in wrapper scripts generated by winegcc
  postFixup = ''
    for exe in "$out"/bin/*.exe; do
      substituteInPlace "$exe" \
        --replace 'WINELOADER="wine"' 'WINELOADER="${wine}/bin/wine"'
    done
  '';

  meta = with lib; {
    description = "Yet Another VST bridge, run Windows VST2 plugins under Linux";
    homepage = "https://github.com/robbert-vdh/yabridge";
    license = licenses.gpl3Plus;
    maintainers = with maintainers; [ kira-bruneau ];
    platforms = [ "x86_64-linux" ];
  };
}
