{ lib, stdenv
, fetchurl
, xz
, writeText
}:

# Note: this package is used for bootstrapping fetchurl, and thus
# cannot use fetchpatch! All mutable patches (generated by GitHub or
# cgit) that are needed here should be included directly in Nixpkgs as
# files.

stdenv.mkDerivation rec {
  pname = "gzip";
  version = "1.11";

  src = fetchurl {
    url = "mirror://gnu/gzip/${pname}-${version}.tar.xz";
    sha256 = "01vrly90rvc98af6rcmrb3gwv1l6pylasvsdka23dffwizb9b6lv";
  };

  outputs = [ "out" "man" "info" ];

  enableParallelBuilding = true;

  nativeBuildInputs = [ xz.bin ];

  makeFlags = [ "SHELL=/bin/sh" "GREP=grep" ];

  # Many gzip executables are shell scripts that depend upon other gzip
  # executables being in $PATH.  Rather than try to re-write all the
  # internal cross-references, just add $out/bin to PATH at the top of
  # all the executables that are shell scripts.
  preFixup = ''
    sed -i '1{;/#!\/bin\/sh/aPATH="'$out'/bin:$PATH"
    }' $out/bin/*
  '';

  # set GZIP env variable to "-n" to stop gzip from adding timestamps
  # to archive headers: https://github.com/NixOS/nixpkgs/issues/86348
  setupHook = writeText "setup-hook" ''
    export GZIP="-n"
  '';

  meta = {
    homepage = "https://www.gnu.org/software/gzip/";
    description = "GNU zip compression program";

    longDescription =
      ''gzip (GNU zip) is a popular data compression program written by
        Jean-loup Gailly for the GNU project.  Mark Adler wrote the
        decompression part.

        We developed this program as a replacement for compress because of
        the Unisys and IBM patents covering the LZW algorithm used by
        compress.  These patents made it impossible for us to use compress,
        and we needed a replacement.  The superior compression ratio of gzip
        is just a bonus.
      '';

    platforms = lib.platforms.all;

    license = lib.licenses.gpl3Plus;
  };
}
