/*
 *  Copyright (C) 2021 Maciej Krüger
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 +  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include<ngx_config.h>
#include<ngx_core.h>
#include<ngx_http.h>
#include<stdlib.h>
#include<stdio.h>

static int keylog_file_fd = -1;

typedef struct {
    ngx_str_t keylog_file;
    ngx_flag_t keylog;
    int keylog_fd;
    ngx_str_t current_keylog_file;
} ngx_keylog_conf_t;

/* Key extraction via the new OpenSSL 1.1.1 API. */
static void keylog_callback(const SSL *ssl, const char *line)
{
    if (keylog_file_fd >= 0) {
        if (write(keylog_file_fd, line, strlen(line))) {
        }
        if (write(keylog_file_fd, "\n", 1)) {
        }
    }
}

#define FIRSTLINE   "# SSL key logfile generated by ngx_http_keylog.c\n"
#define FIRSTLINE_LEN (sizeof(FIRSTLINE) - 1)

static void *
ngx_common_keylog_create_main_conf(ngx_conf_t *cf)
{
    ngx_keylog_conf_t  *conf;
    conf = ngx_pcalloc(cf->pool, sizeof(ngx_keylog_conf_t));
    if (conf == NULL) {
        return NGX_CONF_ERROR;
    }

    conf->keylog = NGX_CONF_UNSET;
    conf->keylog_fd = -1;

    return conf;
}

static char *
ngx_common_keylog_init_main_conf(ngx_conf_t *cf, void *conf) {
    ngx_keylog_conf_t *klcf;
    klcf = conf;

    if (klcf->keylog == 1) {
        if (!klcf->keylog_file.len) {
            ngx_log_error(NGX_LOG_EMERG, cf->log, 0,
                          "keylog enabled but no keylog_file set");
            return NGX_CONF_ERROR;
        }

        // setup keylogging
        if (klcf->current_keylog_file.len != klcf->keylog_file.len || !ngx_strncmp(klcf->keylog_file.data, klcf->current_keylog_file.data, klcf->keylog_file.len)) {
            if (klcf->keylog_fd >= 0) {
                close(klcf->keylog_fd);
                klcf->keylog_fd = -1;
            }

            char * kl = ngx_palloc(cf->pool, klcf->keylog_file.len + 1);
            if (kl == NULL)
                return NGX_CONF_ERROR;
            ngx_memcpy(kl, klcf->keylog_file.data, klcf->keylog_file.len);
            kl[klcf->keylog_file.len] = '\0';
            klcf->keylog_fd = open(kl, O_WRONLY | O_APPEND | O_CREAT, 0644);

            if (klcf->keylog_fd >= 0 && lseek(klcf->keylog_fd, 0, SEEK_END) == 0) {
                /* file is opened successfully and there is no data (pos == 0) */
                if (write(klcf->keylog_fd, FIRSTLINE, FIRSTLINE_LEN)) {
                    return NGX_CONF_ERROR;
                }
            }

            klcf->current_keylog_file.len = klcf->keylog_file.len;
        }

        ngx_log_error(NGX_LOG_WARN, cf->log, 0,
                      "keylog enabled, logging to %*s", klcf->keylog_file.len, klcf->keylog_file.data);
    } else {
        // cleanup fd and current loc
        if (klcf->keylog_fd >= 0) {
            close(klcf->keylog_fd);
            klcf->keylog_fd = -1;
        }
        if (klcf->current_keylog_file.len) {
            ngx_free(klcf->current_keylog_file.data);
            klcf->current_keylog_file.len = 0;
            klcf->current_keylog_file.data = NULL;
        }
    }

    keylog_file_fd = klcf->keylog_fd;

    return NGX_CONF_OK;
}

static void attach_keylog(char* type, ngx_connection_t* connection) {
    if (connection->ssl != NULL && connection->ssl->session_ctx != NULL) {
        ngx_log_error(NGX_LOG_NOTICE, connection->log, 0,
                      "%s keylog on connection %d", type, connection->number);

        SSL_CTX_set_keylog_callback(connection->ssl->session_ctx, keylog_callback);

    }
}
