'use strict'

const crypto = require('crypto')
const fs = require('fs')
const assert = require('assert').strict

const privFormat = {
  type: 'pkcs8', // is a standard for unified private key storage
  format: 'pem', // well-known
  cipher: 'AES-256-CBC' // the better ECB
}

const pubFormat = {
  type: 'spki', // only standard node supports that does both ec and rsa
  format: 'pem' // well-known
}

// TODO: detect keytype and use elgamal for ecc
// ref https://www.npmjs.com/package/eccrypto

class PublicKey {
  constructor (key) {
    assert(key instanceof crypto.KeyObject, 'must be a KeyObject - do you mean PrivateKey.fromXYZ instead?')
    this.key = key
  }

  /* Crypto */

  verify (data, signature) {
    return crypto.verify(
      'sha512',
      Buffer.from(data),
      {
        key: this.key,
        padding: crypto.constants.RSA_PKCS1_PSS_PADDING
      },
      Buffer.from(signature)
    )
  }

  encrypt (data) {
    return crypto.publicEncrypt(
      {
        key: this.key,
        padding: crypto.constants.RSA_PKCS1_OAEP_PADDING,
        oaepHash: 'sha512'
      },
      Buffer.from(data)
    )
  }

  spkiFingerprint (hexString = true) {
    return crypto.createHash('sha512').update(this.export()).digest(hexString ? 'hex' : undefined)
  }

  /* Export */

  exportDER () {
    return this.key.export({
      ...pubFormat,
      format: 'der'
    })
  }

  export () {
    return this.key.export(pubFormat)
  }

  exportToFile (file) {
    fs.writeFileSync(file, this.export())
  }

  /* From */

  static fromKeyObject (key) {
    return new PublicKey(key)
  }

  static fromBuffer (buffer) {
    return PublicKey.fromKeyObject(crypto.createPublicKey({
      key: buffer,
      ...pubFormat
    }))
  }

  static fromDERBuffer (buffer) {
    return PublicKey.fromKeyObject(crypto.createPublicKey({
      key: buffer,
      ...pubFormat,
      format: 'der'
    }))
  }

  static fromFile (file) {
    return PublicKey.fromBuffer(fs.readFileSync(file))
  }
}

class PrivateKey {
  constructor (key, pub) {
    assert(key instanceof crypto.KeyObject, 'must be a KeyObject - do you mean PrivateKey.fromXYZ instead?')
    this.key = key

    if (pub) {
      this.pub = pub
    }
  }

  /* Crypto */

  sign (data) {
    return crypto.sign('sha512', Buffer.from(data), {
      key: this.key,
      padding: crypto.constants.RSA_PKCS1_PSS_PADDING
    })
  }

  decrypt (data) {
    return crypto.privateDecrypt(
      {
        key: this.key,
        padding: crypto.constants.RSA_PKCS1_OAEP_PADDING,
        oaepHash: 'sha512'
      },
      data
    )
  }

  get verify () {
    return this.pub.verify.bind(this.pub)
  }

  get encrypt () {
    return this.pub.encrypt.bind(this.pub)
  }

  get spkiFingerprint () {
    return this.pub.spkiFingerprint.bind(this.pub)
  }

  /* Export */

  export (passphrase) {
    assert(passphrase, 'passphrase is required for key export')

    return this.key.export({
      ...privFormat,
      passphrase
    })
  }

  exportDER (passphrase) {
    assert(passphrase, 'passphrase is required for key export')

    return this.key.export({
      ...privFormat,
      passphrase,
      format: 'der'
    })
  }

  exportToFile (file, passphrase) {
    fs.writeFileSync(file, this.export(passphrase))
  }

  /* From */

  static fromKeyObject (key, pub) {
    return new PrivateKey(key, pub)
  }

  static fromBuffer (buffer, passphrase, pub) {
    return PrivateKey.fromKeyObject(
      crypto.createPrivateKey({
        key: buffer,
        passphrase,
        ...privFormat
      }),
      pub)
  }

  static fromDERBuffer (buffer, passphrase, pub) {
    return PrivateKey.fromKeyObject(
      crypto.createPrivateKey({
        key: buffer,
        passphrase,
        ...privFormat,
        format: 'der'
      }),
      pub)
  }

  static fromFile (file, passphrase, pub) {
    return PrivateKey.fromBuffer(fs.readFileSync(file), passphrase, pub)
  }
}

module.exports = { PublicKey, PrivateKey }
