"use strict";
const eslint_etc_1 = require("eslint-etc");
const utils_1 = require("../utils");
const mutatorRegExp = /^(fill|reverse|sort)$/;
const creatorRegExp = /^(concat|entries|filter|keys|map|slice|splice|values)$/;
const rule = utils_1.ruleCreator({
    defaultOptions: [],
    meta: {
        docs: {
            category: "Possible Errors",
            description: "Forbids the assignment of returned, mutated arrays.",
            recommended: false,
        },
        fixable: undefined,
        messages: {
            forbidden: "Assignment of mutated arrays is forbidden.",
        },
        schema: [],
        type: "problem",
    },
    name: "no-assign-mutated-array",
    create: (context) => {
        const { couldBeType } = eslint_etc_1.getTypeServices(context);
        return {
            [`CallExpression > MemberExpression[property.name=${mutatorRegExp.toString()}]`]: (memberExpression) => {
                const callExpression = eslint_etc_1.getParent(memberExpression);
                const parent = eslint_etc_1.getParent(callExpression);
                if (parent && !eslint_etc_1.isExpressionStatement(parent)) {
                    if (couldBeType(memberExpression.object, "Array") &&
                        mutatesReferencedArray(callExpression)) {
                        context.report({
                            messageId: "forbidden",
                            node: memberExpression.property,
                        });
                    }
                }
            },
        };
        function isNewArray(node) {
            if (eslint_etc_1.isArrayExpression(node)) {
                return true;
            }
            if (eslint_etc_1.isNewExpression(node)) {
                return true;
            }
            if (eslint_etc_1.isCallExpression(node)) {
                const { callee } = node;
                if (eslint_etc_1.isIdentifier(callee) && callee.name === "Array") {
                    return true;
                }
                if (eslint_etc_1.isMemberExpression(callee) &&
                    eslint_etc_1.isIdentifier(callee.object) &&
                    callee.object.name === "Array") {
                    return true;
                }
            }
            return false;
        }
        function mutatesReferencedArray(callExpression) {
            if (eslint_etc_1.isMemberExpression(callExpression.callee)) {
                const memberExpression = callExpression.callee;
                const { object, property } = memberExpression;
                if (eslint_etc_1.isIdentifier(property) && creatorRegExp.test(property.name)) {
                    return false;
                }
                if (isNewArray(object)) {
                    return false;
                }
                if (eslint_etc_1.isCallExpression(object)) {
                    return mutatesReferencedArray(object);
                }
            }
            return true;
        }
    },
});
module.exports = rule;
