"use strict";
const eslint_etc_1 = require("eslint-etc");
const tag_1 = require("../tag");
const tslint_tag_1 = require("../tslint-tag");
const utils_1 = require("../utils");
const deprecatedNamesByProgram = new WeakMap();
const defaultOptions = [];
const rule = utils_1.ruleCreator({
    defaultOptions,
    meta: {
        docs: {
            category: "Best Practices",
            description: "Forbids the use of deprecated APIs.",
            recommended: false,
        },
        fixable: undefined,
        messages: {
            forbidden: `"{{name}}" is deprecated: {{comment}}`,
        },
        schema: [
            {
                properties: {
                    ignored: { type: "object" },
                },
                type: "object",
            },
        ],
        type: "problem",
    },
    name: "no-deprecated",
    create: (context, unused) => {
        const [{ ignored = {} } = {}] = context.options;
        const ignoredNameRegExps = [];
        const ignoredPathRegExps = [];
        Object.entries(ignored).forEach(([key, value]) => {
            switch (value) {
                case "name":
                    ignoredNameRegExps.push(new RegExp(key));
                    break;
                case "path":
                    ignoredPathRegExps.push(new RegExp(key));
                    break;
                default:
                    break;
            }
        });
        const { esTreeNodeToTSNodeMap, program } = eslint_etc_1.getParserServices(context);
        const typeChecker = program.getTypeChecker();
        const getPath = (identifier) => {
            const type = typeChecker.getTypeAtLocation(identifier);
            return typeChecker.getFullyQualifiedName(type.symbol);
        };
        let deprecatedNames = deprecatedNamesByProgram.get(program);
        if (!deprecatedNames) {
            deprecatedNames = tag_1.findTaggedNames("deprecated", program);
            deprecatedNamesByProgram.set(program, deprecatedNames);
        }
        return {
            Identifier: (node) => {
                var _a;
                switch ((_a = eslint_etc_1.getParent(node)) === null || _a === void 0 ? void 0 : _a.type) {
                    case "ExportSpecifier":
                    case "ImportDefaultSpecifier":
                    case "ImportNamespaceSpecifier":
                    case "ImportSpecifier":
                        return;
                    default:
                        break;
                }
                const identifier = esTreeNodeToTSNodeMap.get(node);
                if (!(deprecatedNames === null || deprecatedNames === void 0 ? void 0 : deprecatedNames.has(identifier.text))) {
                    return;
                }
                if (tslint_tag_1.isDeclaration(identifier)) {
                    return;
                }
                if (ignoredNameRegExps.some((regExp) => regExp.test(identifier.text)) ||
                    ignoredPathRegExps.some((regExp) => regExp.test(getPath(identifier)))) {
                    return;
                }
                const tags = tslint_tag_1.getTags("deprecated", identifier, typeChecker);
                if (tags.length > 0) {
                    for (const tag of tags) {
                        context.report({
                            data: {
                                comment: tag.trim().replace(/[\n\r\s\t]+/g, " "),
                                name: identifier.text,
                            },
                            messageId: "forbidden",
                            node,
                        });
                    }
                }
            },
        };
    },
});
module.exports = rule;
