"use strict";
const eslint_etc_1 = require("eslint-etc");
const utils_1 = require("../utils");
const rule = utils_1.ruleCreator({
    defaultOptions: [],
    meta: {
        docs: {
            category: "Best Practices",
            description: "Forbids throwing - or rejecting with - non-`Error` values.",
            recommended: false,
        },
        fixable: undefined,
        messages: {
            forbidden: "{{usage}} non-`Error` values is forbidden.",
        },
        schema: [],
        type: "problem",
    },
    name: "throw-error",
    create: (context) => {
        const sourceCode = context.getSourceCode();
        const { couldBeType, isAny } = eslint_etc_1.getTypeServices(context);
        const checkRejection = (node) => {
            const { arguments: [arg], } = node;
            if (!isAny(arg) && !couldBeType(arg, "Error")) {
                context.report({
                    data: { usage: "Rejecting with" },
                    messageId: "forbidden",
                    node: arg,
                });
            }
        };
        return {
            "CallExpression > MemberExpression[property.name='reject']": (node) => {
                if (!couldBeType(node.object, /^Promise/)) {
                    return;
                }
                checkRejection(eslint_etc_1.getParent(node));
            },
            "NewExpression[callee.name='Promise'] > ArrowFunctionExpression, NewExpression[callee.name='Promise'] > FunctionExpression": (node) => {
                const [, param] = node.params;
                if (!param) {
                    return;
                }
                const text = sourceCode.getText(param);
                const variable = context
                    .getDeclaredVariables(node)
                    .find((variable) => variable.name === text);
                if (!variable) {
                    return;
                }
                variable.references.forEach(({ identifier }) => {
                    const parent = eslint_etc_1.getParent(identifier);
                    if (eslint_etc_1.isCallExpression(parent) && identifier === parent.callee) {
                        checkRejection(parent);
                    }
                });
            },
            ThrowStatement: (node) => {
                if (node.argument &&
                    !isAny(node.argument) &&
                    !couldBeType(node.argument, /^(Error|DOMException)$/)) {
                    context.report({
                        data: { usage: "Throwing" },
                        messageId: "forbidden",
                        node: node.argument,
                    });
                }
            },
        };
    },
});
module.exports = rule;
