'use strict'

const Boom = require('@hapi/boom')
const Joi = require('joi')

module.exports = {
  pkg: require('../package.json'),
  requirements: {
    hapi: '>=18.4.0'
  },
  register: (server, options) => {
    server.auth.scheme('crypto-auth', implementation)
  }
}

const schema = Joi.object({
  tolerance: Joi.number().integer().min(1000).default(5 * 60 * 1000),
  getDevice: Joi.function().required()
})

const headerSchema = Joi.object({
  'x-sig': Joi.string().base64().required(),
  'x-sig-now': Joi.number().required(),
  'x-sig-fingerprint': Joi.string().hex().min(128).max(128).required()
}).unknown(true)

const JSONStableStringify = require('json-stable-stringify')

const delta = (a, b) => a > b ? a - b : b - a

const implementation = (server, options) => {
  const { value: { getDevice, tolerance }, error } = schema.validate(options)
  if (error) {
    throw error
  }

  return {
    authenticate: async (request, h) => {
      const { value, error } = headerSchema.validate(request.headers)

      if (error) {
        throw error
      }

      const { 'x-sig': sig, 'x-sig-now': now, 'x-sig-fingerprint': fingerprint } = value

      const device = await getDevice(fingerprint)

      if (!device) {
        throw Boom.unauthorized('Device unknown')
      }

      const key = device.key

      return h.authenticated({
        credentials: device.credentials ? device.credentials : { fingerprint },
        artifacts: { key, fingerprint, sig, now }
      })
    },
    payload: async (request, h) => {
      const { sig, now, key } = request.auth.artifacts

      const payload = request.payload
      const method = request.method
      const path = request.path

      const toSign = {
        method,
        path,
        now,
        query: []
      }

      for (const [key, value] of new URLSearchParams(request.query).entries()) {
        toSign.query.push([key, value])
      }

      if (payload) {
        toSign.payload = payload
      }

      if (delta(now, Date.now()) > tolerance) {
        throw Boom.unauthorized('Signature invalid')
      }

      if (!key.verify(JSONStableStringify(toSign), Buffer.from(sig, 'base64'))) {
        throw Boom.unauthorized('Signature invalid')
      }

      return h.continue
    }
  }
}
