import {
  getTypes, getType,
  getSTypes, getSType
} from './types.js'

function getId () {
  return 'a' + String(Math.random()).replace(/[^1-6]/g, '').substr(0, 6)
}

async function typeConfig (cliCommon, { id, rid, domain, route }, config) {
  const { prompt, storage, $t } = cliCommon

  for (const key in config) { // eslint-disable-line guard-for-in
    const option = config[key]

    await prompt(
      storage.sub(['domains', id, 'routes', rid, 'config', key]),
      {
        ...option,
        message: $t(option.display)
      }
    )
  }
}

async function addRoute (cliCommon, { id, domain }, { route, skippable }) {
  const { prompt, storage, $t } = cliCommon
  const rid = getId()

  if (!route) {
    route = await prompt(
      storage.sub(['domains', id, 'routes', rid, 'dest']),
      {
        type: 'route',
        message: $t('proxy.domain.add.route')
      }
    )
  }

  const type = await prompt(
    storage.sub(['domains', id, 'routes', rid, 'type']),
    {
      promptType: 'select',
      choices: getTypes().concat(skippable ? ['skip'] : []),
      message: $t('proxy.route.type')
    }
  )

  if (type === 'skip') {
    await storage.del(['domains', id, 'routes', rid, 'type'])
    return
  }

  const typeObj = getType(type)

  await typeConfig(cliCommon, { id, rid, domain, route }, typeObj.config)
}

async function createDomain (cliCommon, { first }) {
  const { prompt, storage, $t } = cliCommon
  const id = getId()

  const domain = await prompt(
    storage.sub(['domains', id, 'name']),
    {
      type: 'domain',
      message: $t('proxy.domain.domainName')
    }
  )

  const useMain = await prompt(
    storage.sub(['domains', id, 'useMain']),
    {
      type: 'bool',
      promptType: 'confirm',
      message: $t('guardian.domain.useMain')
    }
  )

  if (useMain) {
    await addRoute(cliCommon, { id, domain }, { route: '@', skippable: true })
  }

  await addRoute(cliCommon, { id, domain }, { route: true, skippable: true })

  return id
}

async function manageDomain (cliCommon, id) {
  const { prompt, storage, $t } = cliCommon

  const domain = await storage.sub(['domains', id]).get()
  const name = domain.name
  const routes = domain.routes

  const list = []

  for (const id in routes) { // eslint-disable-line guard-for-in
    const route = routes[id]
    list.push({ name: id, message: `${route.dest} => ${route.type}` })
  }

  list.push(
    { name: '#useMain', message: 'useMain toggle' },
    { name: '#addRoute', message: 'add route' },
    { name: '$', message: 'back' }
  )

  const choice = await prompt(
    storage.sub(['_']),
    {
      promptType: 'select',
      choices: list,
      message: $t('guardian.domain.manage.domain', name)
    }
  )

  switch (choice) {
    case '#useMain': {
      await storage.set(['domains', id, 'useMain'], !domain.useMain)
      return manageDomain(cliCommon, id)
    }
    case '#addRoute': {
      await addRoute(cliCommon, { id, domain: name }, {})
      return manageDomain(cliCommon, id)
    }
    case '$': {
      return
    }
    default: {
      const typeObj = getType(domain.routes[choice].type)

      await typeConfig(cliCommon, { id, rid: choice, domain, route: domain.routes[choice].dest }, typeObj.config)

      return manageDomain(cliCommon, id)
    }
  }
}

async function manageServices (cliCommon, domain) {
  const { prompt, storage, $t } = cliCommon

  const services = await storage.sub(['services']).get()

  const list = []

  for (const id in services) { // eslint-disable-line guard-for-in
    const service = services[id]
    list.push({ name: id, message: `${service.type}` })
  }

  list.push(
    { name: '$', message: 'back' }
  )

  const choice = await prompt(
    storage.sub(['_']),
    {
      promptType: 'select',
      choices: list,
      message: $t('guardian.domain.manage.overviewServices')
    }
  )

  switch (choice) {
    case '$': {
      return
    }
    default: {
      await manageService(cliCommon, choice)
    }
  }
}

async function domainOverview (cliCommon) {
  const { prompt, storage, $t } = cliCommon

  const domains = await storage.sub(['domains']).get()

  const list = []

  for (const id in domains) { // eslint-disable-line guard-for-in
    const domain = domains[id]
    list.push({ name: id, message: `${domain.name} (main ${domain.useMain})` })
  }

  list.push(
    { name: '+', message: 'create' },
    { name: '@', message: 'services' },
    { name: '$', message: 'back' }
  )

  const choice = await prompt(
    storage.sub(['_']),
    {
      promptType: 'select',
      choices: list,
      message: $t('guardian.domain.manage.overview')
    }
  )

  switch (choice) {
    case '+': {
      const id = await createDomain(cliCommon, {})
      await manageDomain(cliCommon, id)
      return domainOverview(cliCommon)
    }
    case '@': {
      await manageServices(cliCommon)
      return domainOverview(cliCommon)
    }
    case '$': {
      return
    }
    default: {
      await manageDomain(cliCommon, choice)
      return domainOverview(cliCommon)
    }
  }
}

export default async cliCommon => {
  cliCommon = Object.assign({}, cliCommon, { storage: cliCommon.storage.sub('guardian') })
  const { prompt, storage, $t } = cliCommon

  if (!Object.keys((await storage.sub(['domains']).get()) || {}).length) {
    await createDomain(cliCommon, { first: true })
  }

  await domainOverview(cliCommon)
}
