import dlv from 'dlv'

// TODO: possibly use more extensive lib in the future
import { format as printf } from 'util'
import sharedMessages from './common-locales/index.js'

/*
 * Simple object check.
 *
 * @param item
 * @returns {boolean}
 */
function isObject (item) {
  return (item && typeof item === 'object' && !Array.isArray(item))
}

/*
 * Deep merge two objects.
 *
 * @param target
 * @param ...sources
 * @param {...any} sources
 */
function mergeDeep (target, ...sources) {
  if (!sources.length) return target
  const source = sources.shift()

  if (isObject(target) && isObject(source)) {
    for (const key in source) {
      if (isObject(source[key])) {
        if (!target[key]) Object.assign(target, { [key]: {} })
        mergeDeep(target[key], source[key])
      } else {
        Object.assign(target, { [key]: source[key] })
      }
    }
  }

  return mergeDeep(target, ...sources)
}

// Code copied from linked Stack Overflow question
// https://stackoverflow.com/questions/27936772/how-to-deep-merge-instead-of-shallow-merge

function osLang (env = process.env) {
  return env.LC_ALL || env.LC_MESSAGES || env.LANG || env.LANGUAGE
}

function browserLang () {
  return window.navigator.language
}

/*

locales:
  require('./index.js')
    module.exports = { de: require('./de.json') }

*/

function mergeMessages (base, shared) {
  return Object.keys(base).map(key => [key, mergeDeep(base[key], shared[key] || {})]).reduce((out, next) => {
    out[next[0]] = next[1]
    return out
  }, {})
}

export default function I18N ({ messages, language }) {
  messages = mergeMessages(messages, sharedMessages)

  if (!language) {
    language = (global.window ? browserLang() : osLang()).split(/[-_]/)[0]
  }

  if (!messages[language]) {
    language = 'en'
  }

  function translate (id, ...params) {
    // TODO: printf notation support

    let msg = dlv(messages[language] || {} || {}, id)

    if (!msg) {
      msg = dlv(messages.en, id)
    }

    if (!msg) {
      msg = translate('untranslated', JSON.stringify(Array.isArray(id) ? id.join('.') : id), language)
    }

    return printf(msg, ...params
      .map(p => Array.isArray(p) && typeof p[0] === 'string' ? translate(...p) : p) // nested translations
    )
  }

  translate.extend = newBase => {
    return I18N({ messages: mergeMessages(newBase, messages), language })
  }

  return {
    translate,
    setLanguage: _language => (language = _language),
    getLanguage: () => language
  }
}
