import { IS_DEBUG_VM, spawn } from './util.js'
import yaml from 'js-yaml'

// TODO: schlachtfeld von source code hier mal aufräumen lol - @mkg20001

import fs from 'fs'
import path from 'path'
import os from 'os'

import fetch from 'node-fetch'

const nicPath = (...a) => path.join('/sys/class/net', ...a)
const nicExists = (...a) => fs.existsSync(nicPath(...a))
const nicRead = (...a) => fs.readFileSync(nicPath(...a))

function getNics () {
  return fs.readdirSync('/sys/class/net')
    .filter(net =>
      nicExists(net, 'device') && !nicExists(net, 'wireless')
    )
}

function formatAddr (a) {
  switch (a.family) {
    case 'IPv6': return `[${a.address}]`
    case 'IPv4': return a.address
    default: throw new TypeError(a.family)
  }
}

function getAddrs () {
  const nics = os.networkInterfaces()
  const addrs = getNics().map(nic => nics[nic].filter(a => !a.internal)).flat()
  const urls = addrs.map(addr => `http://${formatAddr(addr)}:3333`)

  return {
    addrs,
    urls
  }
}

async function setupNet (linkConfig, version, storage) {
  const famId = `ipv${version}`

  // prompt "$fam_id" ""

  // Adresse für $fam_name@$_NIC (* angeben um DHCP zu verwenden, - angeben um zu deaktivieren, format: 'ADDRESSE/MASKE')

  // TODO: replace with list, where user can choose "enter address (format: address/netmask)" or "DHCP" or "disable"

  storage = storage.sub(famId)
  const mode = await storage.sub('mode').get()

  switch (mode) {
    case 'static': {
      linkConfig[`dhcp${version}`] = 'no'
      linkConfig[`gateway${version}`] = await storage.sub('gateway').get()

      linkConfig.addresses.push(await storage.sub('address').get())
      linkConfig.nameservers.addresses.push(await storage.sub('dns').get())
      break
    }
    case 'dhcp': {
      linkConfig[`dhcp${version}`] = 'yes'
      break
    }
    case 'disable': {
      linkConfig[`dhcp${version}`] = 'no'
      break
    }
    default: {
      throw new TypeError(mode)
    }
  }
}

async function setupNic (storage, pipeline) {
  if (IS_DEBUG_VM) {
    return
  }

  storage = storage.sub('net')

  const nic = await storage.sub('nic').get()

  const linkConfig = {
    nameservers: {
      addresses: []
    },
    addresses: []
  }

  await setupNet(linkConfig, 4, storage)
  await setupNet(linkConfig, 6, storage)

  /*

  fam_name="IPv$1"
  fam_id="ipv$1"
  prompt "$fam_id" "Adresse für $fam_name@$_NIC (* angeben um DHCP zu verwenden, - angeben um zu deaktivieren, format: 'ADDRESSE/MASKE')"
  if [ "$(_db $fam_id)" == "*" ]; then
    echo "[*] $fam_name@$_NIC wird auf DHCP geschaltet"
    CONF="$CONF
      dhcp$1: yes"
  elif [ "$(_db $fam_id)" == "-" ]; then
    echo "[*] $fam_name@$_NIC wird nicht konfiguriert"
    CONF="$CONF
      dhcp$1: no"
  else
    prompt "$fam_id.gateway" "Gateway für $fam_name"
    prompt "$fam_id.dns" "DNS Server für $fam_name"
    CONF="$CONF
      dhcp$1: no
      gateway$1: $(_db $fam_id.gateway)"
    _ADDR+=("$(_db $fam_id)")
    _DNS+=("$(_db $fam_id.dns)")
    echo "[*] $fam_name@$_NIC wird auf Addresse $(_db $fam_id) geschaltet"
  fi

  */

  // test if config persists

  const netplanConfig = {
    network: {
      version: 2,
      // renderer: 'networkd',
      renderer: 'NetworkManager',
      ethernets: {
        [nic]: linkConfig
      }
    }
  }

  // renderer: networkd
  // service systemd-networkd restart

  fs.writeFileSync('/etc/netplan/50-guardian.yaml', yaml.dump(netplanConfig))
  await spawn('netplan', ['apply'], false, false, {}, pipeline)
}

const exampleAddress = {
  ipv4: '192.168.0.1/24',
  ipv6: 'fe80::c25:5d74:8c72:90d0/64'
}

async function promptNet (version, { storage, prompt, $t }) {
  const famId = `ipv${version}`
  const famName = `IPv${version}`
  const nicName = await storage.sub('nic').get()

  // Adresse für $fam_name@$_NIC (* angeben um DHCP zu verwenden, - angeben um zu deaktivieren, format: 'ADDRESSE/MASKE')

  // TODO: replace with list, where user can choose "enter address (format: address/netmask)" or "DHCP" or "disable"

  const type = await prompt(
    storage.sub([famId, 'mode']),
    {
      promptType: 'select',
      choices: [
        'static', 'dhcp', 'disable'
      ],
      message: $t('common.nicSetup.chooseMode', `${famName}@${nicName}`)
    }
  )

  switch (type) {
    case 'static': {
      await prompt(
        storage.sub([famId, 'address']),
        {
          type: famId + 'cidr',
          message: $t('common.nicSetup.setAddress', famName, nicName, exampleAddress[famId])
        }
      )

      // prompt "$fam_id.gateway" "Gateway für $fam_name"
      await prompt(
        storage.sub([famId, 'gateway']),
        {
          type: famId,
          message: $t('common.nicSetup.setupGateway', famName)
        }
      )

      // prompt "$fam_id.dns" "DNS Server für $fam_name"
      await prompt(
        storage.sub([famId, 'dns']),
        {
          type: 'host',
          message: $t('common.nicSetup.setupDns', famName)
        }
      )
      break
    }
    case 'dhcp':
    case 'disable': {
      break
    }
    default: {
      throw new TypeError(type)
    }
  }
}

async function promptNic (cliCommon) {
  /* if (IS_DEBUG_VM) {
    return
  } */

  cliCommon = Object.assign({}, cliCommon, { storage: cliCommon.storage.sub('net') })

  const { storage, prompt, $t, $b } = cliCommon

  const adapters = await $b.post('/net/getNics')

  // Netzwerkadapter der verwendet werden soll
  await prompt(
    storage.sub('nic'),
    {
      message: $t('common.nicSetup.chooseNic'),
      promptType: 'select',
      choices: adapters
    }
  )

  await promptNet(4, cliCommon)
  await promptNet(6, cliCommon)
}

async function checkNet () {
  try {
    const res = await fetch('https://update.xeredo.it')
    await res.text()
    if (res.status !== 200) {
      throw new Error('Wrong status')
    }

    return { success: true, error: false, display: ['common.net.x_reachable'] }
  } catch (error) {
    return { success: false, error: String(error), display: ['common.net.x_unreachable', String(error)] }
  }
}

export default {
  cli: promptNic,
  sys: {
    // background
    setupNic,
    // util
    getNics,
    getAddrs,
    checkNet
  }
}
