import Boom from '@hapi/boom'
import SessionCookie from './session-cookie.js'
import Joi from 'joi'

import PAM from 'node-linux-pam'
const { pamAuthenticatePromise, pamErrors, PamError } = PAM

export default {
  pkg: {
    name: 'pam-auth',
    version: '1'
  },
  requirements: {
    hapi: '>=18.4.0'
  },
  register: (server, options) => {
    server.auth.scheme('pam-auth', implementation)
  }
}

const schema = Joi.object({
  cookieFolder: Joi.string().required()
})

const implementation = (server, options) => {
  const { value: { cookieFolder }, error } = schema.validate(options)
  if (error) {
    throw error
  }

  const { authenticate, write } = SessionCookie(cookieFolder)

  server.state('xsession', {
    encoding: 'iron',
    password: 'password-should-be-32-characters',
    ignoreErrors: true,
    isSecure: false
  })

  server.route({
    path: '/pam-login',
    method: 'POST',
    config: {
      plugins: {
        licensing: {
          alwaysAllow: true
        }
      },
      validate: {
        payload: Joi.object({
          username: Joi.string().required().min(1),
          password: Joi.string().required().min(1)
        })
      },
      auth: false,
      handler: async (request, h) => {
        let res
        try {
          res = await pamAuthenticatePromise(request.payload)
        } catch (error) {
          if (error instanceof PamError) {
            const { message, code } = error

            throw Boom.unauthorized(message)
          } else {
            throw error
          }
        }

        const xFF = request.headers['x-forwarded-for']
        const ip = xFF ? xFF.split(',')[0].trim() : request.info.remoteAddress

        h.state('xsession', write(request.payload.username, ip))

        return { ok: true, res }
      }
    }
  })

  return {
    authenticate: (request, h) => {
      const cookie = request.state.xsession

      return authenticate(cookie, h)
    }
  }
}
