import * as isIp from 'is-ip'
import isCidr from 'is-cidr'

const D_REG = /^(([a-zA-Z]{1})|([a-zA-Z]{1}[a-zA-Z]{1})|([a-zA-Z]{1}\d{1})|(\d{1}[a-zA-Z]{1})|([a-zA-Z0-9][a-zA-Z0-9-_]{1,61}[a-zA-Z0-9]))\.([a-zA-Z]{2,6}|[a-zA-Z0-9-]{2,30}\.[a-zA-Z]{2,3})$/
const isDomain = str => D_REG.test(String(str))

const e = (trId, ...trParams) => {
  throw Object.assign(new TypeError(trId), {
    trId,
    trParams
  })
}

export default {
  host: {
    u: 'text',
    v: val => {
      if (typeof val !== 'string' || (!isIp(val) && !isDomain(val))) {
        e('common.types.host')
      }

      if (isIp.v6(val)) {
        if (!val.startsWith('[')) {
          val = `[${val}]`
        }
      }

      return val
    }
  },

  ipv4: {
    u: 'text',
    v: val => {
      if (!isIp.v4(val)) {
        e('common.types.ipv4')
      }

      return val
    }
  },

  ipv6: {
    u: 'text',
    v: val => {
      if (!isIp.v6(val)) {
        e('common.types.ipv6')
      }

      return val
    }
  },

  ip: {
    u: 'text',
    v: val => {
      if (!isIp(val)) {
        e('common.types.ip')
      }

      return val
    }
  },

  ipv4cidr: {
    u: 'text',
    v: val => {
      if (!isCidr.v4(val)) {
        e('common.types.ipv4cidr')
      }

      return val
    }
  },

  ipv6cidr: {
    u: 'text',
    v: val => {
      if (!isCidr.v6(val)) {
        e('common.types.ipv6cidr')
      }

      return val
    }
  },

  cidr: {
    u: 'text',
    v: val => {
      if (!isCidr(val)) {
        e('common.types.cidr')
      }

      return val
    }
  },

  path: {
    u: 'text',
    v: val => {
      // TODO: improve

      if (!val.startsWith('/')) {
        e('common.types.path')
      }

      return val
    }
  },

  domain: {
    u: 'text',
    v: val => {
      // TODO: improve

      if (!val) {
        e('common.types.domain')
      }

      return val
    }
  },

  subdomain: {
    u: 'text',
    v: val => {
      // TODO: improve

      if (!val) {
        e('common.types.subdomain')
      }

      return val
    }
  },

  string: {
    u: 'text',
    v: val => {
      if (!val) {
        e('common.types.string')
      }

      return val
    }
  },

  port: {
    u: 'number',
    v: val => {
      if (typeof val === 'string') {
        if (!val.match(/^\d{1,2,3,4,5}$/)) {
          e('common.types.port-noint')
        }

        val = parseInt(val, 10)
      }

      if (val > 2 ** 16 || val < 0) {
        e('common.types.port-range')
      }

      return val
    }
  },

  bool: {
    u: 'checkbox',
    v: val => {
      if (typeof val !== 'boolean') {
        e('common.types.bool')
      }

      return val
    }
  },

  nano: {
    u: 'textfield',
    v: val => {
      return val.trim()
    }
  }
}
