import fetch from 'node-fetch'
import debug from 'debug'

const log = debug('scim')

function validate (data, schema, isCreate = false /* on create some things can be set that otherwise can't be modified */) {
  // TODO: actually validate, throw error if data doesn't match schema
  return true
}

async function listResponse (req, newStartIndex = null) {
  const reqClone = req.clone()
  if (newStartIndex) {
    reqClone.query('startIndex', newStartIndex)
  }

  const res = await reqClone.exec()

  const { itemsPerPage, startIndex, totalResults } = res
  if (itemsPerPage) { // this result has pagination
    if (startIndex - 1 + itemsPerPage < totalResults) { // if we are not at the end
      res.hasNext = true
      res.next = () => listResponse(req, startIndex + itemsPerPage)
    }

    // NOTE: since count isn't specified anywhere fixed there isn't really a way to go backwards, as we don't know what we should set startIndex to, so we just don't
  }

  Object.assign(res, { // TODO: figure out direct assign syntax
    async * [Symbol.asyncIterator] () {
      let r = res

      while (true) {
        if (r.Resources) {
          for (let i = 0; i < r.Resources.length; i++) {
            yield r.Resources[i]
          }
        }

        if (!r.hasNext) {
          break
        }

        r = await r.next()
      }
    }
  })

  res.all = async () => {
    let out = []

    let r = res

    while (true) {
      if (r.Resources) {
        out = out.concat(r.Resources)
      }

      if (!r.hasNext) {
        break
      }

      r = await r.next()
    }

    return {
      Resources: out,
      totalResults: out.length
    }
  }

  return res
}

function reqBuilder (req, self = { method: 'GET', query: {} }) {
  const S = {
    query: (key, value) => {
      self.query[key] = value
      return S
    },
    body: body => {
      self.body = body
      return S
    },
    method: method => {
      self.method = method
      return S
    },
    url: url => {
      self.url = url
      return S
    },
    clone: () => {
      return reqBuilder(req, Object.assign({}, self))
    },
    exec: () => {
      return req(self.url, self.method, self.query, self.body)
    }
  }

  return S
}

function Resource (req, { endpoint, schema }, schemas) {
  return {
    create: data => {
      validate(data, schemas)
      return req().url(`${endpoint}`).method('POST').body(data).exec()
    },
    read: async id => {
      try {
        return await req().url(`${endpoint}/${id}`).exec()
      } catch (error) {
        if (error.status === 404) {
          return
        }

        throw error
      }
    },
    replace: (id, data) => {
      return req().url(`${endpoint}/${id}`).method('PUT').body(data).exec()
    },
    delete: id => {
      return req().url(`${endpoint}/${id}`).method('DELETE').exec()
    },
    update: (id, data) => {
      return req().url(`${endpoint}/${id}`).method('PATCH').body(data).exec()
    },
    get: ({ filter = {}, sortBy, sortOrder, count } = {}) => {
      const r = req().url(`${endpoint}`)

      if (sortBy) {
        r.query('sortBy', sortBy)
      }

      if (sortOrder) {
        r.query('sortOrder', sortOrder)
      }

      if (count) {
        r.query('count', count)
      }

      // TODO: construct and append filter

      return listResponse(r)
    },
    bulk: data => {
      return req().url(`${endpoint}/Bulk`).method('POST').body(data).exec()
    }
  }
}

function Req (base, auth) {
  return () => {
    return reqBuilder(async (url, method = 'GET', query = {}, body) => {
      const headers = {
        authorization: typeof auth === 'function' ? auth() : auth,
        accept: 'application/scim+json'
      }

      log('do request %s %o q=%o b=%o', method, url, query, body)

      if (body) {
        headers['content-type'] = 'application/scim+json'
      }

      const req = await fetch(`${base}${url}?${String(new URLSearchParams(query))}`, {
        method,
        headers,
        body: body ? JSON.stringify(body) : undefined
      })

      const res = await req.json()

      if ((res.schemas && res.schemas[0] === 'urn:ietf:params:scim:api:messages:2.0:Error') || res.error) {
        throw Object.assign(new Error(`[${res.status || req.status}] ${res.detail || res.error}`), { status: req.status, ...res }) // some broken impl return non-spec errors as .error, namely keycloak scim
      }

      return res
    })
  }
}

export default async (base, auth) => {
  const req = Req(base, auth)

  const serviceProviderConfig = await (await listResponse(req().url('/ServiceProviderConfig'))).all()
  const schemas = await (await listResponse(req().url('/Schemas'))).all()
  const resourceTypes = await (await listResponse(req().url('/ResourceTypes'))).all()

  const out = {
    schemas,
    resourceTypes,
    serviceProviderConfig,
    resources: {}
  }

  resourceTypes.Resources.forEach(res => {
    out.resources[res.id] = Resource(req, res, schemas)
  })

  return out
}
