import Hapi from '@hapi/hapi'
import Boom from '@hapi/boom'

import SPA from 'hapi-spa-serve'
import path from 'path'

import Core from './core.js'

import Cookie from '@xeredo/common/src/modules/session-cookie.js'
import ENV from './loadenv.js'

/* eslint-disable guard-for-in */
/* eslint-disable no-loop-func */

import { fileURLToPath } from 'url'

async function require (thing) {
  const imported = await import(thing) // "dynamic import"
  return Object.assign(imported.default, imported)
}
const __dirname = path.dirname(fileURLToPath(import.meta.url))

const Relish = (await require('relish2'))({
  messages: {}
})

const COOKIE_FOLDER = '/var/lib/guardian/.cli-cookie'

export default async config => {
  config.hapi.routes = {
    validate: {
      failAction: Relish.failAction
    }
  }

  const server = Hapi.server(config.hapi)
  const cookie = Cookie(COOKIE_FOLDER)

  await server.register({
    plugin: await require('@xeredo/common/src/modules/cli-auth.js')
  })

  await server.register({
    plugin: await require('@xeredo/common/src/modules/pam-auth.js')
  })

  await server.register({
    plugin: await require('@xeredo/common/src/common/hapi/multiauth.js')
  })

  await server.register({
    plugin: await require('@xeredo/licensing-client/src/hapi.js'),
    options: {
      storage: '/var/lib/guardian/.license',
      server: ENV.LICENSING.SERVER,
      bcert: ENV.LICENSING.BCERT,
      /* server: 'http://localhost:12012',
      bcert: path.join(__dirname, '..', 'src', 'licensing-client', 'bcert.dev'), */
      requiredFeatures: ['guardian.base']
    }
  })

  server.auth.strategy('cli-session', 'cli-auth', {
    cookieFolder: COOKIE_FOLDER
  })

  server.auth.strategy('pam-session', 'pam-auth', {
    cookieFolder: COOKIE_FOLDER
  })

  server.auth.strategy('multi', 'multiauth', {
    strategies: ['cli-session', 'pam-session']
  })

  server.auth.default('multi')

  await server.route({
    path: '/self',
    method: 'GET',
    config: {
      plugins: {
        licensing: {
          alwaysAllow: true
        }
      },
      auth: {
        mode: 'try'
      },
      handler (request, h) {
        const { isAuthenticated } = request.auth

        if (!isAuthenticated) {
          throw Boom.unauthorized()
        }

        return {
          licenseOk: server.methods.licenseOk(),
          features: server.methods.getCurrentFeatures(),
          ...request.auth.artifacts
        }
      }
    }
  })

  await server.route({
    path: '/auth/logout',
    method: 'GET',
    config: {
      plugins: {
        licensing: {
          alwaysAllow: true
        }
      },
      auth: {
        mode: 'try'
      },
      handler (request, h) {
        const { isAuthenticated, credentials } = request.auth

        if (isAuthenticated) {
          cookie.revoke(credentials.cookie)
        }

        return h.redirect('/')
      }
    }
  })

  await server.register({
    plugin: await require('hapi-pino'),
    options: {
      name: ''
    }
  })

  await server.register({
    plugin: await require('@hapi/nes'),
    options: {}
  })

  if (global.SENTRY) {
    await server.register({
      plugin: await require('hapi-sentry'),
      options: { client: global.SENTRY }
    })
  }

  await server.register({
    plugin: await require('@hapi/inert'),
    options: {}
  })

  server.route({
    method: 'GET',
    path: '/docs/{param*}',
    config: {
      plugins: {
        licensing: {
          alwaysAllow: true
        }
      },
      auth: false,
      handler: {
        directory: {
          path: path.join(__dirname, '../../dist/docs'),
          index: ['index.html'],
          redirectToSlash: true
        }
      }
    }
  })

  SPA(server, {
    assets: path.join(__dirname, '../../dist'),
    options: {
      plugins: {
        licensing: {
          alwaysAllow: true
        }
      },
      auth: false
    }
  })

  // main logic

  await Core(server, config)

  async function stop () {
    await server.stop()
  }

  await server.start()

  process.on('SIGINT', () => {
    stop()
  })

  process.on('SIGTERM', () => {
    stop()
  })
}
