import Boom from '@hapi/boom'
import Licensing from './index.js'
import dlv from 'dlv'
import Joi from 'joi'

export default {
  pkg: {
    name: 'licensing',
    version: '1'
  },
  requirements: {
    hapi: '>=18.4.0'
  },
  register: (server, options) => {
    const licensing = Licensing(options)

    server.ext('onPreHandler', function (request, h) {
      if (request.route.path.startsWith('/nes/')) {
        return h.continue // always allow NES internals, as we can't explicitly allow them
      }

      if (dlv(request, 'route.settings.plugins.licensing.alwaysAllow', false)) { // always allow
        return h.continue
      }

      if (licensing.isOK()) {
        return h.continue
      }

      throw Boom.paymentRequired('No valid license')
    })

    server.method('hasLicense', () => licensing.hasLicense())
    server.method('getCurrentLicenses', () => licensing.getCurrentLicenses())
    server.method('getCurrentFeatures', () => licensing.getCurrentFeatures())
    server.method('licenseOk', () => licensing.isOK())
    server.method('signedRequest', (...a) => licensing.request(...a))
    server.method('signRequest', (...a) => licensing.signRequest(...a))

    server.route({
      path: '/license/redeem',
      method: 'POST',
      config: {
        plugins: {
          licensing: {
            alwaysAllow: true
          }
        },
        validate: {
          payload: Joi.object({
            code: Joi.string().uuid().required()
          })
        },
        handler: async function (request, h) {
          try {
            await licensing.redeem(request.payload.code, options.requiredFeatures)
          } catch (error) {
            return Boom.badRequest('Invalid license or other error')
          }

          return { ok: true }
        }
      }
    })

    server.route({
      path: '/license',
      method: 'GET',
      config: {
        plugins: {
          licensing: {
            alwaysAllow: true
          }
        },
        handler: async function (request, h) {
          return {
            ok: licensing.isOK(),
            hasLicense: licensing.hasLicense(),
            features: licensing.getCurrentFeatures(),
            licenses: licensing.getCurrentLicenses()
          }
        }
      }
    })
  }
}
