import _mkdirp from 'mkdirp'
import Crypto from './crypto.js'
import fetch from 'node-fetch'
import path from 'path'
import { fileURLToPath } from 'url'

const __dirname = path.dirname(fileURLToPath(import.meta.url))
const { sync: mkdirp } = _mkdirp

export default ({
  server = 'https://licensing.xeredo.it',
  tlsCertPin = false,
  storage = '/etc/.license',
  bcert = path.join(__dirname, 'bcert'),
  requiredFeatures = []
} = {}) => {
  mkdirp(storage)

  const prefix = '/api/v0/license'

  const crypto = Crypto({ storage, bcert })

  let lastOK = Date.now()

  async function request (url, method, payload) {
    const req = await fetch(...crypto.signedRequest(url, method, payload))
    const res = await req.json()

    return res
  }

  async function doShutdown () {
    // send sigterm for shutdown trigger
    process.kill(process.pid, 'SIGTERM')

    // if that fails/times out, do exit(2)
    setTimeout(() => process.exit(2), 60 * 1000).unref()
  }

  function getCurrentLicenses () {
    return crypto.lcert ? crypto.lcert.data.licenses : []
  }

  function getCurrentFeatures () {
    return crypto.lcert
      ? crypto.lcert.data.features.reduce((out, next) => {
        out[next.key] = JSON.parse(next.value)

        return out
      }, {})
      : {}
  }

  async function ping () {
    const res = await request(`${server}${prefix}/ping`, 'GET')

    crypto.processLCert(res.lcert)
  }

  setInterval(ping, 24 * 60 * 60 * 1000).unref()

  function isOK () {
    const features = getCurrentFeatures()

    return Boolean(
      crypto.lcert &&
      !requiredFeatures.filter(feature => !features[feature]).length &&
      crypto.lcert.expiry > Date.now()
    )
  }

  async function checkOK () {
    if (isOK()) {
      // if we have an lcert and all the valid features are present and lcert is not expired, set last ok
      lastOK = Date.now()
    } else if (lastOK - Date.now() > 1000 * 60 * 60 * (crypto.lcert ? 24 * 7 : 3)) {
      // else check when last lastOK was and shutdown
      // if more than 3h ago (or one week if we used to have a license), then do shutdown
      return doShutdown()
    }
  }

  setInterval(checkOK, 60 * 1000).unref()

  return {
    isOK,
    redeem: async (code, requiredFlags) => {
      const res = await fetch(`${server}${prefix}/redeem`, {
        method: 'POST',
        headers: {
          'content-type': 'application/json'
        },
        body: JSON.stringify({
          key: crypto.self.pub.export().toString('base64'),
          code,
          sig: crypto.self.sign(code).toString('base64'),
          requiredFlags
        })
      })

      const json = await res.json()

      if (!json.ok) {
        throw new Error(json.error)
      }

      crypto.processLCert(json.lcert)

      return json
    },
    getCurrentFeatures,
    getCurrentLicenses,
    hasLicense () {
      return Boolean(crypto.lcert)
    },
    request,
    signRequest: crypto.signedRequest
  }
}

export const BCERT = {
  PROD: path.join(__dirname, 'bcert'),
  STAGE: path.join(__dirname, 'bcert.stage'),
  DEV: path.join(__dirname, '..', 'bcert.dev')
}
