import { v4 as uuid } from 'uuid'
import { NGINX_FOLDER, reload } from '../config/apply.js'
import { writeFileSync, unlinkSync, existsSync, createReadStream } from 'fs'
import { join } from 'path'
import Boom from '@hapi/boom'
import { createInterface } from 'readline'
import { spawn } from 'child_process'
import { PassThrough, Readable } from 'stream'

function unlinkSafe (file) {
  if (existsSync(file)) {
    unlinkSync(file)
  }
}

function lineStream (file, handler) {
  const rl = createInterface({
    input: createReadStream(file)
  })
  rl.on('line', handler)
}

const keylogFolder = '/tmp/keylog'

let tcpdumpInstance

const keylogTemplate = type => `
keylog on;
keylog_file ${keylogFolder}/${type};
`

async function enableKeylog (storage) {
  const token = uuid()
  await storage.sub(['intercept', 'keylog']).set({
    token
  })
  writeFileSync(join(NGINX_FOLDER, 'sites', '_keylog.conf'), keylogTemplate('http'))
  writeFileSync(join(NGINX_FOLDER, 'stream.d', '_keylog.conf'), keylogTemplate('stream'))
  await reload()
}

async function disableKeylog (storage) {
  await storage.sub(['intercept', 'keylog']).del()
  await unlinkSafe(join(NGINX_FOLDER, 'sites', '_keylog.conf'))
  await unlinkSafe(join(NGINX_FOLDER, 'stream.d', '_keylog.conf'))
  await reload()
  await unlinkSafe(join(keylogFolder, 'http'))
  await unlinkSafe(join(keylogFolder, 'stream'))
}

async function getKeylogInfo (storage) {
  const config = await storage.sub(['intercept', 'keylog']).get()
  if (!existsSync(join(NGINX_FOLDER, 'sites', '_keylog.conf')) || !config) {
    return { enabled: false }
  }

  return {
    enabled: true,
    token: config.token
  }
}

async function getKeylogStream (storage, providedToken) {
  const config = await storage.sub(['intercept', 'keylog']).get()
  if (!config || config.token !== providedToken) {
    throw Boom.unauthorized('Invalid token')
  }

  const stream = new Readable()
  lineStream(join(keylogFolder, 'http'), line => stream.push(line + '\n'))
  lineStream(join(keylogFolder, 'stream'), line => stream.push(line + '\n'))

  return stream
}

async function enablePCAP (storage) {
  const token = uuid()
  await storage.sub(['intercept', 'pcap']).set({
    token
  })

  if (!tcpdumpInstance) {
    // const p = spawn('tcpdump', ['-i', 'any', '-w', '/proc/self/fd/1'], { stdio: ['pipe', 'pipe', process.stderr] })
    const p = spawn('sh', ['-c', 'cat | tcpdump -i any -w /dev/stdout | cat'], { stdio: ['pipe', 'pipe', process.stderr] })
    // prevent cache buildup, we don't want to store anything in ram
    p.stdout.uncork()
    p.stdout.resume()

    p.once('exit', () => {
      tcpdumpInstance = null
      disablePCAP(storage)
    })

    tcpdumpInstance = {
      clear () {
        p.kill()
      },
      stream () {
        return p.stdout.pipe(new PassThrough())
      }
    }
  }
}

async function disablePCAP (storage) {
  if (tcpdumpInstance) {
    tcpdumpInstance.clear() // this triggers disablePCAP() again async later, so the code needs to be idempotent
  }
  await storage.sub(['intercept', 'pcap']).del()
}

async function getPCAPInfo (storage) {
  const config = await storage.sub(['intercept', 'pcap']).get()
  if (!config || !tcpdumpInstance) {
    return { enabled: false }
  }

  return {
    enabled: true,
    token: config.token,
    port: tcpdumpInstance.port
  }
}

async function getPCAPStream (storage, providedToken) {
  const config = await storage.sub(['intercept', 'pcap']).get()
  if (!config || config.token !== providedToken || !tcpdumpInstance) {
    throw Boom.unauthorized('Invalid token')
  }

  return tcpdumpInstance.stream()
}

export default {
  sys: {
    enableKeylog,
    disableKeylog,
    getKeylogInfo,
    getKeylogStream,
    enablePCAP,
    disablePCAP,
    getPCAPInfo,
    getPCAPStream
  }
}
