import fetch from 'node-fetch'
import path from 'path'
import fs from 'fs'
import os from 'os'
import debug from 'debug'
import _mkdirp from 'mkdirp'
import _rimraf from 'rimraf'

import { spawn } from '@xeredo/common/src/modules/util.js'
import { reload } from '../config/apply.js'
import _util from '../config/util.js'

import glob from 'glob'

const { sync: mkdirp } = _mkdirp
const { sync: rimraf } = _rimraf

const { fromTemplate } = _util({})

const read = (...p) => String(fs.readFileSync(path.join(...p)))

const prom = f => new Promise((resolve, reject) => f((err, res) => err ? reject(err) : resolve(res)))

const log = debug('guardian:waf')

const FINAL = '/var/lib/guardian/modsec'
const FINAL_DATA_RW = '/var/lib/guardian/modsec_data_rw'
const WAF_CONF = '/etc/nginx/conf.d/03-waf.conf'

const EXTRA_RULES = `

## Extra Rules from Guardian

SecDataDir ${FINAL_DATA_RW}
SecRuleEngine On
# SecDefaultAction "phase:1,deny,log"

SecDebugLog /var/log/nginx/modsec_debug.log
SecDebugLogLevel 3

SecAuditLog /var/log/nginx/modsec_audit.log
SecAuditEngine RelevantOnly
SecAuditLogRelevantStatus "^(?:5|4(?!04))"
SecAuditLogParts ABIJDEFHZ
`

const wafs = {
  owasp: {
    name: 'waf.list.owasp',
    desc: 'waf.list.owasp_desc',
    async checkForUpdate () {
      let releases = await fetch('https://api.github.com/repos/coreruleset/coreruleset/releases', {
        Accept: 'application/vnd.github.v3+json'
      })
      releases = await releases.json()
      const r = releases.filter(r => !r.prerelease && !r.draft)[0]

      return {
        version: r.tag_name,
        params: {
          dl: r.tarball_url
        }
      }
    },
    async download (params, output, tmp, final) { // final is not where the plugin should write, final is where it should expect files to be
      log('[owasp] fetching %s', params.dl)
      const res = await fetch(params.dl)
      const targz = path.join(tmp, 'owasp.tar.gz')
      const targzStream = fs.createWriteStream(targz)

      await prom(cb => {
        res.body.once('error', cb)
        targzStream.once('error', cb)
        targzStream.once('finish', () => cb())
        res.body.pipe(targzStream)
      })

      log('[owasp] extracting')

      await spawn('tar', ['xzf', targz, '-C', tmp], false, false)

      const g = ['*/crs-setup.conf.example', '*/rules/*.conf', '*/rules/*.data']
      const files = (await Promise.all(g.map(p => path.join(tmp, p)).map(p =>
        prom(cb =>
          glob(p, cb)
        )))).flat()

      const outconf = fs.createWriteStream(path.join(output, 'owasp.conf'))

      outconf.write(EXTRA_RULES)

      files.forEach(f => {
        if (f.endsWith('.conf') || f.endsWith('.example')) { // append .conf
          outconf.write(`## ${path.relative(tmp, f)}\n\n`)
          outconf.write(read(f) + '\n')
        } else if (f.endsWith('.data')) { // copy .data
          fs.writeFileSync(path.join(output, path.basename(f)), fs.readFileSync(f))
        }
      })

      outconf.close()

      await prom(cb => outconf.once('close', () => cb()))

      return {
        main: 'owasp.conf'
      }
    }
  }
}

async function setupWAF (storage, pipeline) {
  storage = storage.sub('guardian')

  const waf = (await storage.sub('waf').get()) || {}

  log('waf setup %o', waf)

  if (waf.enabled === false) {
    // we don't delete the waf stuff (FINAL), so when user re-enables quickly afterwards
    // (for testing if error caused by WAF) stuff is still there and it is instant switch
    log('disabling waf')
    rimraf(WAF_CONF)
    await reload()
    return
  }

  const type = waf.type || 'owasp'

  const set = wafs[type]

  const { version, params } = await set.checkForUpdate()

  let confParameters = waf.confParameters

  log('current %s@%s - new %s@%s', waf.current, waf.currentVersion, type, version)

  if (!waf.current || waf.current !== type || waf.currentVersion !== version) {
    log('need waf update...')
    const out = path.join(os.tmpdir(), String(Math.random()))
    mkdirp(out)
    const tmp = path.join(os.tmpdir(), String(Math.random()))
    mkdirp(tmp)
    log('downloading waf')
    const { extra = '', main } = await set.download(params, out, tmp, FINAL)
    rimraf(tmp)

    // finished, now switch

    log('switching waf')

    confParameters = {
      FILE: path.join(FINAL, main),
      EXTRA: extra
    }
    await (storage.sub(['waf', 'confParameters']).set(confParameters))

    rimraf(FINAL)
    fs.renameSync(out, FINAL)

    mkdirp(FINAL_DATA_RW)
    await spawn('chown', ['nginx:nginx', FINAL_DATA_RW, '-R'])

    // clear old links
    fs.readdirSync(FINAL_DATA_RW)
      .map(f => path.join(FINAL_DATA_RW, f))
      .filter(f => fs.lstatSync(f).isSymbolicLink() && !fs.existsSync(f))
      .forEach(f => rimraf(f))

    // make new links
    fs.readdirSync(FINAL)
      .filter(f => f.endsWith('.data'))
      .map(f => [path.join(FINAL, f), path.join(FINAL_DATA_RW, f)])
      .filter(([from, to]) => !fs.existsSync(to))
      .forEach(([from, to]) => fs.symlinkSync(from, to))

    await (storage.sub(['waf', 'current']).set(type))
    await (storage.sub(['waf', 'currentVersion']).set(version))
    log('waf update successfull')
  }

  // TODO: better system for safely reloading nginx

  // we can't handle this in the update routine, because we don't unset current type when disabling (cache for temp off)
  // so we just redo config + reload each time
  log('applying waf')
  fs.writeFileSync(WAF_CONF, fromTemplate('03-waf.conf', confParameters))
  await reload()
}

async function getWAFList (storage) {
  const s = (await (storage.sub(['guardian', 'waf']).get())) || {}
  return {
    current: {
      id: s.current,
      enabled: s.enabled === false ? s.enabled : true,
      version: s.currentVersion,
      newID: s.type || 'owasp' // if this doesn't match type it means waf change hasn't been applied yet
    },
    list: Object.keys(wafs).map(k => ({
      id: k,
      name: wafs[k].name,
      desc: wafs[k].desc
    }))
  }
}

export default {
  // cli: promptWAF,
  sys: {
    // background
    setupWAF,
    // foreground
    getWAFList
  }
}
