#! /usr/bin/env nix-shell
#! nix-shell -i bash -p nix curl jq nix-prefetch-github git gnused gnugrep -I nixpkgs=.

set -eu -o pipefail

tmpfile=$(mktemp "update-stackage.XXXXXXX")
# shellcheck disable=SC2064

stackage_config="pkgs/development/haskell-modules/configuration-hackage2nix/stackage.yaml"

trap "rm ${tmpfile} ${tmpfile}.new" 0
touch "$tmpfile" "$tmpfile.new" # Creating files here so that trap creates no errors.

curl -L -s "https://stackage.org/lts/cabal.config" >"$tmpfile"
old_version=$(grep "# Stackage" $stackage_config | sed -E 's/.*([0-9]{2}\.[0-9]+)/\1/')
version=$(sed -rn "s/^--.*http:..(www.)?stackage.org.snapshot.lts-//p" "$tmpfile")

if [[ "$old_version" == "$version" ]]; then
   echo "No new stackage version"
   exit 0 # Nothing to do
fi

echo "Updating Stackage LTS from $old_version to $version."

# Create a simple yaml version of the file.
sed -r \
    -e '/^--/d' \
    -e 's|^constraints:||' \
    -e 's|^ +|  - |' \
    -e 's|,$||' \
    -e '/installed$/d' \
    -e '/^$/d' \
    < "${tmpfile}" | sort --ignore-case >"${tmpfile}.new"

cat > $stackage_config << EOF
# Stackage LTS $version
# This file is auto-generated by
# maintainers/scripts/haskell/update-stackage.sh
default-package-overrides:
EOF

# Drop restrictions on some tools where we always want the latest version.
sed -r \
    -e '/ cabal2nix /d' \
    -e '/ distribution-nixpkgs /d' \
    -e '/ jailbreak-cabal /d' \
    -e '/ language-nix /d' \
    < "${tmpfile}.new" >> $stackage_config

if [[ "${1:-}" == "--do-commit" ]]; then
git add $stackage_config
git commit -F - << EOF
haskellPackages: stackage-lts $old_version -> $version

This commit has been generated by maintainers/scripts/haskell/update-stackage.sh
EOF
fi
