{ config, lib, pkgs, ... }:

with lib;

let
  pkg = pkgs.haste-server;
  cfg = config.services.haste-server;

  format = pkgs.formats.json {};
in
{
  options.services.haste-server = {
    enable = mkEnableOption "haste-server";
    openFirewall = mkEnableOption "firewall passthrough for haste-server";

    settings = mkOption {
      description = ''
        Configuration for haste-server.
        For documentation see <link xlink:href="https://github.com/toptal/haste-server#settings">project readme</link>
      '';
      type = format.type;
    };
  };

  config = mkIf (cfg.enable) {
    networking.firewall.allowedTCPPorts = mkIf (cfg.openFirewall) [ cfg.settings.port ];

    services.haste-server = {
      settings = mkDefault {
        host = "::";
        port = 7777;

        keyLength = 10;
        maxLength = 400000;

        staticMaxAge = 86400;
        recompressStaticAssets = false;

        logging = [
          {
            level = "verbose";
            type = "Console";
            colorize = true;
          }
        ];

        keyGenerator = {
          type = "phonetic";
        };

        rateLimits = {
          categories = {
            normal = {
              totalRequests = 500;
              every = 60000;
            };
          };
        };

        storage = {
          type = "file";
        };

        documents = {
          about = "${pkg}/share/haste-server/about.md";
        };
      };
    };

    systemd.services.haste-server = {
      wantedBy = [ "multi-user.target" ];
      requires = [ "network.target" ];
      after = [ "network.target" ];

      serviceConfig = {
        User = "haste-server";
        DynamicUser = true;
        StateDirectory = "haste-server";
        WorkingDirectory = "/var/lib/haste-server";
        ExecStart = "${pkg}/bin/haste-server ${format.generate "config.json" cfg.settings}";
      };

      path = with pkgs; [ pkg coreutils ];
    };
  };
}
