{ lib, stdenv, fetchFromGitHub
, autoreconfHook
, glibcLocales, kmod, coreutils, perl
, dmidecode, hwdata, sqlite
, nixosTests
}:

stdenv.mkDerivation rec {
  pname = "rasdaemon";
  version = "0.6.7";

  src = fetchFromGitHub {
    owner = "mchehab";
    repo = "rasdaemon";
    rev = "v${version}";
    sha256 = "sha256-vyUDwqDe+HD4mka6smdQuVSM5U9uMv/TrfHkyqVJMIo=";
  };

  nativeBuildInputs = [ autoreconfHook ];

  buildInputs = [
    coreutils
    glibcLocales
    hwdata
    kmod
    sqlite
    (perl.withPackages (ps: with ps; [ DBI DBDSQLite ]))
  ]
  ++ lib.optionals (!stdenv.isAarch64) [ dmidecode ];

  configureFlags = [
    "--sysconfdir=/etc"
    "--localstatedir=/var"
    "--with-sysconfdefdir=${placeholder "out"}/etc/sysconfig"
    "--enable-sqlite3"
    "--enable-aer"
    "--enable-mce"
    "--enable-extlog"
    "--enable-non-standard"
    "--enable-abrt-report"
    "--enable-hisi-ns-decode"
    "--enable-devlink"
    "--enable-diskerror"
    "--enable-memory-failure"
    "--enable-memory-ce-pfa"
    "--enable-amp-ns-decode"
  ]
  ++ lib.optionals (stdenv.isAarch64) [ "--enable-arm" ];

  # The installation attempts to create the following directories:
  # /var/lib/rasdaemon
  #   location of the RAS event log generated by rasdaemon -r
  # /etc/ras/dimm_labels.d
  #   location of the DIMM labels generated by ras-mc-ctl
  # /etc/sysconfig/rasdaemon
  #   location of rasdaemon config file, currently only used for CE PFA config

  # these are optional (for logging, DIMM label storage and user config)
  # /var/lib/rasdaemon should be created by the NixOS module
  # /etc/ras/dimm_labels.d should probably be generated,
  # from user supplied content, in the NixOS module
  # /etc/sysconfig/rasdaemon should be generated if there is user supplied content
  # and default to $out/etc/sysconfig/rasdaemon which should hold the supplied default

  # therefore, stripping these from the generated Makefile
  # (needed in the config flags because those set where the tools look for these)

# easy way out, ends up installing /nix/store/...rasdaemon/bin in $out

  postConfigure = ''
    substituteInPlace Makefile \
      --replace '"$(DESTDIR)/etc/ras/dimm_labels.d"' '"$(prefix)/etc/ras/dimm_labels.d"'
  '';

  outputs = [ "out" "dev" "man" "inject" ];

  postInstall = ''
    install -Dm 0755 contrib/edac-fake-inject $inject/bin/edac-fake-inject
    install -Dm 0755 contrib/edac-tests $inject/bin/edac-tests
  '';

  postFixup = ''
    # Fix dmidecode and modprobe paths
    substituteInPlace $out/bin/ras-mc-ctl \
      --replace 'find_prog ("modprobe")  or exit (1)' '"${kmod}/bin/modprobe"'
  ''
  + lib.optionalString (!stdenv.isAarch64) ''
    substituteInPlace $out/bin/ras-mc-ctl \
      --replace 'find_prog ("dmidecode")' '"${dmidecode}/bin/dmidecode"'
  '';

  passthru.tests = nixosTests.rasdaemon;

  meta = with lib; {
    description = ''
      A Reliability, Availability and Serviceability (RAS) logging tool using EDAC kernel tracing events
    '';
    longDescription = ''
      Rasdaemon is a RAS (Reliability, Availability and Serviceability) logging
      tool. It records memory errors, using the EDAC tracing events. EDAC is a
      Linux kernel subsystem with handles detection of ECC errors from memory
      controllers for most chipsets on i386 and x86_64 architectures. EDAC
      drivers for other architectures like arm also exists.
    '';
    homepage = "https://github.com/mchehab/rasdaemon";
    license = licenses.gpl2Plus;
    platforms = platforms.linux;
    changelog = "https://github.com/mchehab/rasdaemon/blob/v${version}/ChangeLog";
    maintainers = with maintainers; [ evils ];
  };
}
