{ lib
, multiStdenv
, fetchFromGitHub
, fetchpatch
, substituteAll
, pkgsi686Linux
, libnotify
, meson
, ninja
, pkg-config
, wine
, boost
, libxcb
, nix-update-script
}:

let
  # Derived from subprojects/bitsery.wrap
  bitsery = fetchFromGitHub {
    owner = "fraillt";
    repo = "bitsery";
    rev = "c0fc083c9de805e5825d7553507569febf6a6f93";
    sha256 = "sha256-VwzVtxt+E/SVcxqIJw8BKPO2q7bu/hkhY+nB7FHrZpY=";
  };

  # Derived from subprojects/function2.wrap
  function2 = fetchFromGitHub {
    owner = "Naios";
    repo = "function2";
    rev = "02ca99831de59c7c3a4b834789260253cace0ced";
    sha256 = "sha256-wrt+fCcM6YD4ZRZYvqqB+fNakCNmltdPZKlNkPLtgMs=";
  };

  # Derived from subprojects/tomlplusplus.wrap
  tomlplusplus = fetchFromGitHub {
    owner = "marzer";
    repo = "tomlplusplus";
    rev = "8e669aa6990e0ed219c169d491472d749f54c393";
    sha256 = "sha256-l8ckbCqjz3GUfwStcl3H2C+un5dZfT2uLtayvdu93D4=";
  };

  # Derived from vst3.wrap
  vst3 = fetchFromGitHub {
    owner = "robbert-vdh";
    repo = "vst3sdk";
    rev = "v3.7.4_build_25-patched";
    fetchSubmodules = true;
    sha256 = "sha256-oHRJZItw+he5M+beVZkUrhJir6rgFZ80ORzA73mJT2A=";
  };
in multiStdenv.mkDerivation rec {
  pname = "yabridge";
  version = "3.8.0";

  # NOTE: Also update yabridgectl's cargoHash when this is updated
  src = fetchFromGitHub {
    owner = "robbert-vdh";
    repo = pname;
    rev = version;
    sha256 = "sha256-XacJjHxsp60/l36pFPGonUyOsyFF2lmqplAaisHXZDY=";
  };

  # Unpack subproject sources
  postUnpack = ''(
    cd "$sourceRoot/subprojects"
    cp -R --no-preserve=mode,ownership ${bitsery} bitsery
    cp packagefiles/bitsery/* bitsery
    cp -R --no-preserve=mode,ownership ${function2} function2
    cp packagefiles/function2/* function2
    cp -R --no-preserve=mode,ownership ${tomlplusplus} tomlplusplus
    cp -R --no-preserve=mode,ownership ${vst3} vst3
  )'';

  patches = [
    # Hard code bitbridge & runtime dependencies
    (substituteAll {
      src = ./hardcode-dependencies.patch;
      boost32 = pkgsi686Linux.boost;
      libxcb32 = pkgsi686Linux.xorg.libxcb;
      inherit libnotify wine;
    })
    # Remove with next yabridge update
   (fetchpatch {
      name = "fix-for-wine-7.1.patch";
      url = "https://github.com/robbert-vdh/yabridge/commit/de470d345ab206b08f6d4a147b6af1d285a4211f.patch";
      sha256 = "sha256-xJx1zvxD+DIjbkm7Ovoy4RaAvjx936/j/7AYUPh/kOo=";
      includes = [ "src/wine-host/xdnd-proxy.cpp" ];
    })

  ];

  postPatch = ''
    patchShebangs .
  '';

  nativeBuildInputs = [
    meson
    ninja
    pkg-config
    wine
  ];

  buildInputs = [
    boost
    libxcb
  ];

  # Meson is no longer able to pick up Boost automatically.
  # https://github.com/NixOS/nixpkgs/issues/86131
  BOOST_INCLUDEDIR = "${lib.getDev boost}/include";
  BOOST_LIBRARYDIR = "${lib.getLib boost}/lib";

  mesonFlags = [
    "--cross-file" "cross-wine.conf"
    "-Dwith-bitbridge=true"

    # Requires CMake and is unnecessary
    "-Dtomlplusplus:generate_cmake_config=false"
  ];

  installPhase = ''
    runHook preInstall
    mkdir -p "$out/bin" "$out/lib"
    cp yabridge-group*.exe{,.so} "$out/bin"
    cp yabridge-host*.exe{,.so} "$out/bin"
    cp libyabridge-vst2.so "$out/lib"
    cp libyabridge-vst3.so "$out/lib"
    runHook postInstall
  '';

  # Hard code wine path in wrapper scripts generated by winegcc
  postFixup = ''
    for exe in "$out"/bin/*.exe; do
      substituteInPlace "$exe" \
        --replace 'WINELOADER="wine"' 'WINELOADER="${wine}/bin/wine"'
    done
  '';

  passthru.updateScript = nix-update-script {
    attrPath = pname;
  };

  meta = with lib; {
    description = "Yet Another VST bridge, run Windows VST2 plugins under Linux";
    homepage = "https://github.com/robbert-vdh/yabridge";
    license = licenses.gpl3Plus;
    maintainers = with maintainers; [ kira-bruneau ];
    platforms = [ "x86_64-linux" ];
  };
}
