#!/nix/store/5kfq0gsyrzh3491wwxc9mp9sgrqq4b70-nodejs-17.5.0/bin/node

import colors from 'colors/safe.js'
import yargs from 'yargs'
import { hideBin } from 'yargs/helpers'
import { checkAndPull, installUpdate } from './index.js'
import { processUpdate, rebuild } from './core.js'
import { e, r, p, ETC, STORE, error, die, info } from './util.js'

import { dirname } from 'path'
import { fileURLToPath } from 'url'

import pkg2 from 'rimraf'

import KV from './kv.js'

const __dirname = dirname(fileURLToPath(import.meta.url))

/*

Interface:
- xup switch-channel <new-channel>
- xup update
- xup check-update
- xup status
- xup auto-update enable/disable

- xup extra switch-unsafe <new-channel> <new-type>
- xup extra init <nar> <yaml>
- xup extra cron

*/

if (!e(ETC, 'xup.json')) {
  error('Error: /etc/xup.json not found - not a xup-managed system')
  error()
  error('This command is to be exclusively ran on Xeredo Updater managed systems')
  error('and will not be of much use outside of that environment')
  error()
  error('If you belive this to be an error, notify support')
  process.exit(2) // eslint-disable-line no-process-exit
}

const { url, type, buildId, version } = JSON.parse(r(ETC, 'xup.json'))
const settings = KV(p(STORE, 'settings.json'))
const autoUpdate = settings.get('autoUpdate', 2) // 0=off, 1=pull-only, 2=pull and install
const autoUpdateModeHuman = {
  0: 'disabled',
  1: 'download',
  2: 'download and install'
}

const { sync: rimraf } = pkg2

async function updateRoutine (url, type) {
  if (await checkAndPull(url, buildId)) {
    await installUpdate(p(STORE, 'update.yml'), p(STORE, 'update.nar'), version, type)

    info('cleaning up...')

    rimraf(p(STORE, 'available'))
    rimraf(p(STORE, 'update.nar'))
    rimraf(p(STORE, 'update.yml'))
  }
}

yargs(hideBin(process.argv)) // eslint-disable-line no-unused-expressions
  .scriptName('xup')
  .command('switch-channel <new-channel>', 'switch to another channel', yargs => yargs, async argv => {
    await updateRoutine(argv.newChannel, type)
  })
  .command('update', 'check for updates and install them', yargs => yargs, async argv => {
    await updateRoutine(url, type)
  })
  .command('check-update', 'check for updates', yargs => yargs, async argv => {
    await checkAndPull(url, buildId, true)
  })
  .command('status', 'print current status', yargs => yargs, async argv => {
    info('xup version %s', JSON.parse(r(__dirname, '..', 'package.json')).version)
    info('auto update mode: %s', autoUpdateModeHuman[autoUpdate])
    info('installed version: %s', version)
    info('installed type: %s', type)
    info('installed channel URL: %s', url)
    info('installed build id: %s', buildId)
    info('cached available version, if any: %s', e(STORE, 'available') ? r(STORE, 'available') : '(none)')
  })
  .command('rebuild', 'rebuild system', yargs => yargs, async argv => {
    await rebuild()
  })
  /*
    xup version xy
    auto update enabled: yes/no
    current version:
    current type:
    current channel URL:
    installed os version:
    cached available version, if any:
  */
  .command('auto-update <mode>', 'set auto-update mode: ' + Object.entries(autoUpdateModeHuman).map((key, value) => `${key}=${value}`).join(', '), yargs => yargs
    .positional('mode', {
      describe: 'auto-update mode'
    }), argv => {
    if (!autoUpdateModeHuman[argv.mode]) {
      return die('invalid auto-update mode %s', argv.mode)
    }

    settings.set('autoUpdate', argv.mode)
  })
  .command('extra', 'extra commands (don\'t use unless instructed to)', yargs => yargs
    .command('switch-unsafe <new-channel> <new-type>', 'switch to new channel while ignoring any restrictions', yargs => yargs, async argv => {
      await updateRoutine(argv.newChannel, argv.newType)
    })
    .command('init <nar> <yaml>', 'init system', yargs => yargs, async argv => {
      await processUpdate(argv.yaml, argv.nar, type, true)
    })
    .command('cron', 'daily update routine', yargs => yargs, async () => {
      info('auto update mode: %s', autoUpdateModeHuman[autoUpdate])
      switch (autoUpdate) {
        case 0: {
          return info('doing nothing')
        }

        case 1: {
          return checkAndPull(url, buildId)
        }

        case 2: {
          return updateRoutine(url, type)
        }

        default: {
          throw new TypeError('Invalid autoupdate mode: ' + JSON.stringify(autoUpdate))
        }
      }
    })
  )
  .option('colors', {
    alias: 'c',
    type: 'boolean',
    description: 'Enable colors in output',
    default: true,
    coerce: v => v ? colors.enable() : colors.disable()
  })
  .demandCommand(1)
  .help()
  .parse()
